<?php declare(strict_types=1);

namespace EshopCheckout\Model\Builders;

use Contributte\Translation\Translator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Templating\Template;
use Core\Model\Templating\TemplateFactory;
use EshopCatalog\AdminModule\Model\Sellers;
use EshopCatalog\Model\Entities\Seller;
use EshopCheckout\Model\Events\OrderReceiptRenderEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;

class OrderReceiptBuilder
{
	protected Translator $translator;
	protected Order $order;
	protected EventDispatcher $eventDispatcher;
	protected ?Seller $seller;
	protected TemplateFactory $templateFactory;

	public function __construct(Order $order, Translator $translator, EventDispatcher $eventDispatcher,
								Sellers $sellers, TemplateFactory $templateFactory)
	{
		$this->order = $order;
		$this->translator = $translator;
		$this->eventDispatcher = $eventDispatcher;
		$this->templateFactory = $templateFactory;
		$this->seller = $sellers->getSellerForSite($order->site->getIdent());
	}

	public function build(): Template
	{
		// latte
		$customTemplateFile = TEMPLATES_DIR . '/Front/default/EshopCheckout/Receipt/receipt.latte';
		$defaultTemplateFile = __DIR__ . '/../../Templates/Receipt/receipt.latte';

		// build template
		/** @var Template $template */
		$template = $this->templateFactory->createTemplate();
		$template->setFile(file_exists($customTemplateFile) ? $customTemplateFile : $defaultTemplateFile);
		$template->order = $this->order;
		$template->seller = $this->seller;
		$template->title = $this->getTitle();

		$this->eventDispatcher->dispatch(new OrderReceiptRenderEvent($template, $this->order), 'eshopCheckout.beforeOrderReceiptRender');

		return $template;
	}

	protected function getTitle(): string
	{
		return $this->translator->translate('eshopCheckout.receiptTitle', null, ['number' => EshopOrdersConfig::load('receipt.enableUseNumericalSeries', false) ? $this->order->receiptIdent : $this->order->getId()]);
	}

}