<?php declare(strict_types=1);

namespace EshopCheckout\Model\Subscribers;

use Contributte\EventDispatcher\EventDispatcher;
use Core\AdminModule\Model\Sites;
use Core\Model\Entities\EntityManagerDecorator;
use EshopCheckout\Model\Entities\Transaction;
use EshopCheckout\Model\EshopCheckoutConfig;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Orders;
use Nette\Application\LinkGenerator;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderFormSubscriber implements EventSubscriberInterface
{
	protected LinkGenerator $linkGenerator;
	protected EventDispatcher $eventDispatcher;
	protected Orders $orders;
	protected Sites $sites;
	protected const SIGNAL_KEY = 'showReceipt';
	protected EntityManagerDecorator $em;

	public function __construct(LinkGenerator $linkGenerator, Orders $orders, EventDispatcher $eventDispatcher, Sites $sites, EntityManagerDecorator $em)
	{
		$this->linkGenerator = $linkGenerator;
		$this->orders = $orders;
		$this->eventDispatcher = $eventDispatcher;
		$this->sites = $sites;
		$this->em = $em;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.admin.onBeforeRenderEditOrderNavigation' => ['onRenderNavigation', 100],
			'eshopOrders.admin.ordersDefaultNavigationCallback'   => ['onRenderNavigationCallback', 100],
			'eshopCheckout.orderBeforeSave'                       => ['orderBeforeSave', 100]
		];
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		if ($defaultOrderSite = EshopCheckoutConfig::load('defaultOrderSite')) {
			$site = $this->sites->get($defaultOrderSite);
			if ($site && $event->order) {
				$event->order->site = $site;
			}
		}

		if (EshopCheckoutConfig::load('allowNexgoTerminal')) {
			$transactionId = $event->formData['transactionId'];
			if ($transactionId && $event->order && !$event->order->getId()) {
				$transaction = new Transaction($event->order, $transactionId);
				$this->em->persist($transaction);
			}
		}
	}

	public function onRenderNavigation(OrdersNavigationEvent $event): void
	{
		if ($event->order && in_array($event->order->getPaymentIdent(), ['storeCash', 'storeCard'])) {
			$event->addNavigation(
				'eshopCheckout.navigation.bill',
				['navigationCallback!', ['key' => self::SIGNAL_KEY]],
				'fas fa-file-invoice-dollar',
				'ajax bg-success',
				['EshopCheckout:Admin' => 'access']
			);
		}
	}

	public function onRenderNavigationCallback(OrderNavigationCallbackEvent $event): void
	{
		$orderId = (int) $event->presenter->getParameter('id');

		if ($event->key !== self::SIGNAL_KEY) {
			return;
		}

		$event->presenter->redirect(':EshopCheckout:Admin:OrderReceipts:Receipt', ['orderId' => $orderId]);
	}

}