<?php declare(strict_types = 1);

namespace EshopCheckout\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use Apitte\Negotiation\Http\ArrayEntity;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Caster;
use EshopCheckout\ApiModule\Api\V1\Model\Terminals;
use EshopCheckout\Model\Entities\Terminal;
use Exception;
use Tracy\Debugger;

#[Path('/terminals')]
class TerminalsController extends BaseController
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected Terminals              $terminals,
	)
	{
	}

	#[Path('/save')]
	#[Method('POST')]
	public function save(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$data = (array) $request->getJsonBody();

		try {
			$this->em->getConnection()->beginTransaction();

			$terminal = null;
			if ($data['id']) {
				$terminal = $this->em->getReference(Terminal::class, Caster::toInt($data['id']));
			}

			/** @var Terminal $terminal */
			$terminal               = $terminal ?: new Terminal($data['ip'], $data['port'], $data['secureString'], $data['terminalId']);
			$terminal->ip           = Caster::toString($data['ip']);
			$terminal->port         = Caster::toString($data['port']);
			$terminal->secureString = Caster::toString($data['secureString']);
			$terminal->terminalId   = Caster::toString($data['terminalId']);

			$this->em->persist($terminal);
			$this->em->flush();

			$this->em->getConnection()->commit();
		} catch (Exception $e) {
			Debugger::log($e);
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->getConnection()->rollBack();
			}

			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
				'status' => 'error',
				'data'   => [],
			]));
		}

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
			'status' => 'ok',
			'data'   => [
				'id'           => $terminal->getId(),
				'ip'           => $terminal->ip,
				'port'         => $terminal->port,
				'secureString' => $terminal->secureString,
				'isActive'     => $terminal->isActive,
				'terminalId'   => $terminal->terminalId,
			],
		]));
	}

	#[Path('/get-all')]
	#[Method('GET')]
	public function getAll(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$result = $this->terminals->getAllAssoc();

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($result));
	}

	#[Path('/set-active-terminal')]
	#[Method('POST')]
	public function setActiveTerminal(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$data       = (array) $request->getJsonBody();
		$terminalId = Caster::toInt($data['terminalId']);

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
			'status' => $this->terminals->setActive($terminalId) ? 'ok' : 'error',
		]));
	}

	#[Path('/remove')]
	#[Method('POST')]
	public function remove(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$data       = (array) $request->getJsonBody();
		$terminalId = Caster::toInt($data['terminalId']);

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
			'status' => $this->terminals->remove($terminalId) ? 'ok' : 'error',
		]));
	}

}
