<?php declare(strict_types = 1);

namespace EshopCheckout\Model\Builders;

use Contributte\Translation\Translator;
use Core\Model\Entities\Site;
use Core\Model\Event\EventDispatcher;
use Core\Model\Templating\Template;
use Core\Model\Templating\TemplateFactory;
use EshopCatalog\AdminModule\Model\Sellers;
use EshopCatalog\Model\Entities\Seller;
use EshopCheckout\Model\Events\OrderReceiptRenderEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;
use InvalidArgumentException;

class OrderReceiptBuilder
{
	protected ?Seller $seller;

	public function __construct(
		protected Order           $order,
		protected Translator      $translator,
		protected EventDispatcher $eventDispatcher,
		protected Sellers         $sellers,
		protected TemplateFactory $templateFactory
	)
	{
		$seller = $this->order->site instanceof Site ? $sellers->getSellerForSite($this->order->site->getIdent()) : null;

		if (!$seller instanceof Seller) {
			throw new InvalidArgumentException('Seller not found');
		}

		$this->seller = $seller;
	}

	public function build(): Template
	{
		// latte
		$customTemplateFile  = TEMPLATES_DIR . '/Front/default/EshopCheckout/Receipt/receipt.latte';
		$defaultTemplateFile = __DIR__ . '/../../Templates/Receipt/receipt.latte';

		// build template
		/** @var Template $template */
		$template = $this->templateFactory->createTemplate();
		$template->setFile(file_exists($customTemplateFile) ? $customTemplateFile : $defaultTemplateFile);
		$template->order  = $this->order;
		$template->seller = $this->seller;
		$template->title  = $this->getTitle();

		$this->eventDispatcher->dispatch(new OrderReceiptRenderEvent($template, $this->order), 'eshopCheckout.beforeOrderReceiptRender');

		return $template;
	}

	protected function getTitle(): string
	{
		return $this->translator->translate('eshopCheckout.receiptTitle', null, ['number' => EshopOrdersConfig::load('receipt.enableUseNumericalSeries', false) ? $this->order->receiptIdent : $this->order->getId()]);
	}

}
