<?php declare(strict_types = 1);

namespace EshopCheckout\Model\Subscribers;

use Core\AdminModule\Model\Sites;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use EshopCheckout\Model\Entities\Transaction;
use EshopCheckout\Model\EshopCheckoutConfig;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Orders;
use Nette\Application\LinkGenerator;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderFormSubscriber implements EventSubscriberInterface
{
	protected const SIGNAL_KEY = 'showReceipt';

	public function __construct(
		protected LinkGenerator          $linkGenerator,
		protected Orders                 $orders,
		protected EventDispatcher        $eventDispatcher,
		protected Sites                  $sites,
		protected EntityManagerDecorator $em
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.admin.onBeforeRenderEditOrderNavigation' => ['onRenderNavigation', 100],
			'eshopOrders.admin.ordersDefaultNavigationCallback'   => ['onRenderNavigationCallback', 100],
			'eshopCheckout.orderBeforeSave'                       => ['orderBeforeSave', 100],
		];
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		if ($defaultOrderSite = EshopCheckoutConfig::loadString('defaultOrderSite')) {
			$site = $this->sites->get($defaultOrderSite);
			if ($site) {
				$event->order->site = $site;
			}
		}

		if (EshopCheckoutConfig::load('allowNexgoTerminal')) {
			$transactionId = $event->formData['transactionId'];
			if ($transactionId && !$event->order->getId()) {
				$transaction = new Transaction($event->order, $transactionId);
				$this->em->persist($transaction);
			}
		}
	}

	public function onRenderNavigation(OrdersNavigationEvent $event): void
	{
		if ($event->order && in_array($event->order->getPaymentIdent(), ['storeCash', 'storeCard'])) {
			$event->addNavigation(
				'eshopCheckout.navigation.bill',
				['navigationCallback!', ['key' => self::SIGNAL_KEY]],
				'fas fa-file-invoice-dollar',
				'ajax bg-success',
				['EshopCheckout:Admin' => 'access']
			);
		}
	}

	public function onRenderNavigationCallback(OrderNavigationCallbackEvent $event): void
	{
		$orderId = $event->presenter->getParameterInt('id');

		if ($event->key !== self::SIGNAL_KEY || $orderId === null) {
			return;
		}

		$event->presenter->redirect(':EshopCheckout:Admin:OrderReceipts:Receipt', ['orderId' => $orderId]);
	}

}
