import axios from 'axios'
import {authString} from "./api";

export default class NexgoTerminalApi {
	constructor(ip, port, secureString) {
		this.ip = ip
		this.port = port
		this.secureString = secureString
	}

	_url() {
		return `http://${this.ip}:${this.port}`
	}

	async log(data) {
		try {
			await axios.post('/api/v1/eshoporders/nexgo-terminal-log/log', data, {
				headers: {
					'Authorization': authString,
				}
			})
		} catch (e) {
			console.log(e)
		}
	}

	async isNexgoTerminalOnline() {
		try {
			const method = 'isNexgoTerminalOnline'
			const response = await axios.post(`${this._url()}/paya/terminal`, {
					'secureString': this.secureString,
					'checkOnline': true
				}, {timeout: 3500, timeoutErrorMessage: 'timeout'}
			).catch(async error => {
				await this.log({method, response: error})
			})

			if (response.status !== 200) {
				await this.log({method, response})
				return false
			}

			if (response.data.isOnline === 2) {
				await this.log({method, response: response.data})
				return false
			}

			return true

		} catch (e) {
			if (e.code === 'ECONNABORTED') {
				return false
			} else {
				throw e
			}
		}
	}

	async createPaymentNexgoTerminal(transactionId, amount) {
		global.paymentCancelToken = axios.CancelToken.source()
		const method = 'createPaymentNexgoTerminal'
		const response = await axios.post(`${this._url()}/paya/payment`, {
				'secureString': this.secureString,
				'request': {
					'amount': amount * 100,
					'currencyCode': '203',
					'invoiceNumber': '',
					'transactionId': transactionId,
					'daughterCompanyId': ''
				}
			}, global.paymentCancelToken?.token ? {cancelToken: global.paymentCancelToken.token} : {}
		).catch(async error => {
			await this.log({method, response: error})
			throw error
		})

		if (response.status === 202) {
			await this.log({method, response})
		} else if (response.data.status.toLowerCase() !== 'ok') {
			await this.log({method, response: response.data})
		}

		return response
	}

	async checkPaymentStatusNexgoTerminal(transactionId) {
		const method = 'checkPaymentStatusNexgoTerminal'
		const response = await axios.post(`${this._url()}/paya/status`, {
				'transactionId': transactionId,
				'secureString': this.secureString
			}
		).catch(async error => {
			await this.log({method, response: error})
		})

		if (response.data.status !== 'Finished') {
			await this.log({method, response: response.data})
		}

		return response
	}

	async getPaymentResult(transactionId) {
		const method = 'getPaymentResult'
		const response = await axios.post(`${this._url()}/paya/result`, {
				'transactionId': transactionId,
				'secureString': this.secureString
			}
		).catch(async error => {
			await this.log({method, response: error})
		})

		if (response.data.responseCode.toLowerCase() !== 'ok') {
			await this.log({method, response: response.data})
		}

		return response
	}

	async confirmPayment(transactionId) {
		const method = 'confirmPayment'
		const response = await axios.post(`${this._url()}/paya/confirm`, {
				'transactionId': transactionId,
				'secureString': this.secureString,
				'confirm': true
			}
		).catch(async error => {
			await this.log({method, response: error})
		})

		if (!response.data.isConfirmed) {
			await this.log({method, response: response.data})
		}

		return response
	}

	async getInfo() {
		const method = 'getInfo'
		const response = await axios.get(`${this._url()}/paya/info`).catch(async error => {
			await this.log({method, response: error})
		})

		if (response.status !== 200) {
			await this.log({method, response: response.data})
		}

		return response
	}

}

// STARA VERZE

// import axios from 'axios'
//
// const nexgoTerminalConf = {
// 	currencyCode: '203',
// 	daughterCompanyId: ''
// }
//
// export default class NexgoTerminalApi
// {
// 	constructor(ip, port, secureString) {
// 		this.ip = ip
// 		this.port = port
// 		this.secureString = secureString
// 	}
//
// 	_url() {
// 		return `http://${this.ip}:${this.port}`
// 	}
//
// 	async isNexgoTerminalOnline()
// 	{
// 		let isOnline = true
//
// 		try {
// 			const response = await axios.post(`${this._url()}/paya/terminal`, {
// 					'secureString': this.secureString,
// 					'checkOnline': true
// 				}, {timeout: 3500, timeoutErrorMessage: 'timeout'}
// 			)
//
// 			if (response.status !== 200) {
// 				return false
// 			}
//
// 			if (response.data.isOnline === 2) {
// 				return false
// 			}
//
// 			return true
//
// 		} catch (e) {
// 			if (e.code === 'ECONNABORTED') {
// 				return false
// 			} else {
// 				throw e
// 			}
// 		}
// 	}
//
// 	async createPaymentNexgoTerminal(transactionId, amount) {
// 		return await axios.post(`${this._url()}/paya/payment`, {
// 				'secureString': this.secureString,
// 				'request': {
// 					'amount': amount * 100,
// 					'currencyCode': '203',
// 					'invoiceNumber': '',
// 					'transactionId': transactionId,
// 					'daughterCompanyId': ''
// 				}
// 			}
// 		)
// 	}
//
// 	async checkPaymentStatusNexgoTerminal(transactionId) {
// 		return await axios.post(`${this._url()}/paya/status`, {
// 				'transactionId': transactionId,
// 				'secureString': this.secureString
// 			}
// 		)
// 	}
//
// 	async getPaymentResult(transactionId) {
// 		return await axios.post(`${this._url()}/paya/result`, {
// 				'transactionId': transactionId,
// 				'secureString': this.secureString
// 			}
// 		)
// 	}
//
// 	async confirmPayment(transactionId) {
// 		return await axios.post(`${this._url()}/paya/confirm`, {
// 				'transactionId': transactionId,
// 				'secureString': this.secureString,
// 				'confirm': true
// 			}
// 		)
// 	}
//
// 	async getInfo() {
// 		return await axios.get(`${this._url()}/paya/info`)
// 	}
//
// }