<?php declare(strict_types = 1);

namespace EshopGifts\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopCatalog\Model\Entities\Category;
use EshopGifts\AdminModule\Model\CategoryGifts;
use EshopGifts\Model\Entities\CategoryGift;

class CategoryGiftsGrid extends BaseControl
{
	/** @var CategoryGifts */
	protected $categoryGifts;

	/** @var ICategoryGiftFormFactory */
	protected $formFactory;

	public function __construct(ICategoryGiftFormFactory $formFactory,
	                            CategoryGifts $categoryGifts)
	{
		$this->formFactory   = $formFactory;
		$this->categoryGifts = $categoryGifts;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Handle
	 */

	public function handleEdit(int $id): void
	{
		$this->template->modalTitle = $this->t('eshopGifts.title.editCategoryGift');
		$this['form']->setGift($id);
		$this->template->modal            = 'categoryGiftForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$p = $this->getPresenter();
		if ($this->categoryGifts->remove($id)) {
			$p->flashMessageSuccess('default.removed');
			$this['grid']->reload();
		} else
			$p->flashMessageDanger('default.removeFailed');

		$p->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->categoryGifts->getEr()->createQueryBuilder('cg')
			->addSelect('gift, giftT, cats, catsT')
			->innerJoin('cg.gift', 'gift')
			->innerJoin('gift.productTexts', 'giftT', 'with', 'giftT.lang = :lang')
			->leftJoin('cg.categories', 'cats')
			->leftJoin('cats.categoryTexts', 'catsT', 'with', 'catsT.lang = :lang')
			->setParameter('lang', $this->translator->getLocale());

		$grid->setDataSource($qb);

		// Columns
		//TODO obrázek
		$grid->addColumnText('product', 'eshopGifts.gift.name', 'gift.productText.name');
		$grid->addColumnText('code1', 'eshopGifts.gift.code1', 'gift.code1');
		$grid->addColumnText('ean', 'eshopGifts.gift.ean', 'gift.ean');
		$grid->addColumnText('categories', 'eshopGifts.gift.categories')->setRenderer(function(CategoryGift $row) {
			return implode(', ', array_map(function(Category $v) {
				return $v->getCategoryText()->getName();
			}, $row->getCategories()->toArray()));
		});
		$grid->addColumnPrice('priceFrom', 'eshopGifts.gift.priceFrom');
		$grid->addColumnPrice('priceTo', 'eshopGifts.gift.priceTo');
		$grid->addColumnDateTime('dateFrom', 'eshopGifts.gift.dateFrom')->setFormat('j. n. Y');
		$grid->addColumnDateTime('dateTo', 'eshopGifts.gift.dateTo')->setFormat('j. n. Y');

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->addClass('ajax')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	protected function createComponentForm(): CategoryGiftForm
	{
		$control = $this->formFactory->create();

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[] = function() {
				$this->getPresenter()->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->onSuccessSaveAndClose[] = function() {
				$this->getPresenter()->payload->hideModal = true;
				$this->getPresenter()->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']['saveControl']->closeModalOnCancel();
		};

		return $control;
	}
}
