<?php declare(strict_types = 1);

namespace EshopGifts\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\Products;
use EshopGifts\AdminModule\Model\CategoryGifts;
use EshopGifts\Model\Entities\CategoryGift;
use Nette\Application\ForbiddenRequestException;
use Nette\Utils\ArrayHash;

class CategoryGiftForm extends BaseControl
{
	/** @var int @persistent */
	public $giftId;

	/** @var CategoryGift */
	protected $gift;

	/** @var CategoryGifts */
	protected $categoryGifts;

	/** @var Products */
	protected $productsService;

	/** @var Categories */
	protected $categoriesService;

	public function __construct(CategoryGifts $categoryGifts, Products $products, Categories $categories)
	{
		$this->categoryGifts     = $categoryGifts;
		$this->productsService   = $products;
		$this->categoriesService = $categories;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Handler
	 */

	public function handleProductsLoad(): array
	{
		$output = [];

		foreach ($this->productsService->getEr()->createQueryBuilder('p')
			         ->select('p.id, p.code1, p.ean, pt.name')
			         ->innerJoin('p.productTexts', 'pt', 'with', 'pt.lang = :lang')
			         ->setParameters([
				         'lang' => $this->translator->getLocale(),
			         ])->getQuery()->getArrayResult() as $row) {
			$output[$row['id']] = [
				'id'    => $row['id'],
				'name'  => $row['name'],
				'code1' => $row['code1'],
				'ean'   => $row['ean'],
			];
		}

		$this->getPresenter()->sendJson($output);
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->getElementPrototype()->setAttribute('id', 'eshopGifts_categoryGiftForm');
		$form->addHidden('productId');

		$form->addText('product', 'eshopGifts.gift.product')
			->setAttribute('id', 'categoryGiftForm_product')
			->setAttribute('data-link', $this->link('productsLoad!'))
			->setRequired();
		$form->addText('name', 'eshopGifts.gift.name')
			->setMaxLength(255)
			->setNullable();
		$form->addEditor('description', 'eshopGifts.gift.description')
			->setMaxLength(255)
			->setToolbar('Text')
			->setDisableAutoP(false)
			->setHeight(100)
			->setNullable();
		$form->addBool('isActive', 'eshopGifts.gift.isActive')->setDefaultValue(1);
		$form->addCheckboxNestedList('categories', 'eshopGifts.gift.categories', $this->categoriesService->getFlatTree());
		$form->addText('priceFrom', 'eshopGifts.gift.priceFrom')
			->setAttribute('min', 0)
			->setNullable()
			->addRule(BaseForm::FLOAT);
		$form->addText('priceTo', 'eshopGifts.gift.priceTo')
			->setAttribute('min', 0)
			->setNullable()
			->addRule(BaseForm::FLOAT);

		$form->addDatePicker('dateFrom', 'eshopGifts.gift.dateFrom');
		$form->addDatePicker('dateTo', 'eshopGifts.gift.dateTo');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$product = $this->productsService->getReference($values->productId);

			$gift = $this->getGift() ?: new CategoryGift($product);
			$gift->setProduct($product);
			$gift->setCategories($values->categories
				? array_map(function($id) { return $this->categoriesService->getReference($id); }, $values->categories)
				: []);
			$gift->setActive($values->isActive);
			$gift->name        = $values->name;
			$gift->description = $values->description;
			$gift->priceFrom   = $values->priceFrom;
			$gift->priceTo     = $values->priceTo;
			$gift->dateFrom    = $values->dateFrom;
			$gift->dateTo      = $values->dateTo;

			$this->em->persist($gift)->flush();

			$this->getPresenter()->flashMessageSuccess($this->t('eshopGifts.gift.saved'));
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ======================== GET / SET
	 */

	public function getGift(): ?CategoryGift
	{
		if ($this->giftId && !$this->gift)
			$this->gift = $this->categoryGifts->get($this->giftId);

		return $this->gift;
	}

	public function setGift(int $id)
	{
		$gift = $this->categoryGifts->get($id);

		if (!$gift)
			throw new ForbiddenRequestException("Category gift '$id' not found");

		$this->gift   = $gift;
		$this->giftId = $id;

		$categories = [];
		foreach ($gift->getCategories()->toArray() as $cat) {
			if (array_key_exists($cat->getId(), $this['form']['categories']->getItems()))
				$categories[] = $cat->getId();
		}

		$this['form']->setDefaults([
			'productId'   => $gift->getProduct()->getId(),
			'product'     => $gift->getProduct()->code1 . ' | ' . $gift->getProduct()->getProductText()->name,
			'name'        => $gift->name,
			'description' => $gift->description,
			'categories'  => $categories,
			'isActive'    => $gift->isActive(),
			'priceFrom'   => $gift->priceFrom,
			'priceTo'     => $gift->priceTo,
			'dateFrom'    => $gift->dateFrom,
			'dateTo'      => $gift->dateTo,
		]);
	}
}
