<?php declare(strict_types = 1);

namespace EshopGifts\FrontModule\Model\Subscribers;

use Core\Model\Event\DaoEvent;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Event\AfterFillDaoProductEvent;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopGifts\FrontModule\Model\CategoryGifts;
use EshopGifts\FrontModule\Model\Dao\Gift;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao\CartItemGift;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class SetGiftsToDaoSubscriber implements EventSubscriberInterface
{
	/** @var CategoryGifts */
	protected $categoryGifts;

	/** @var array */
	protected $cCategoryGifts;

	/** @var ProductsFacade */
	protected $productsFacade;

	/** @var Carts */
	protected $cartsService;

	/** @var array */
	protected $loadedFor = [];

	public function __construct(CategoryGifts $categoryGifts, ProductsFacade $productsFacade, Carts $carts)
	{
		$this->categoryGifts  = $categoryGifts;
		$this->productsFacade = $productsFacade;
		$this->cartsService   = $carts;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			Products::class . '::afterFillDao' => 'afterFillDaoProduct',
			'eshopOrders.cartFillDaoItems'     => ['fillDaoCartItem', 90],
		];
	}

	public function afterFillDaoProduct(DaoEvent $event): void
	{
		/** @var Product $dao */
		$dao = &$event->dao;

		if (!$dao->categoryGiftsAllowed) {
			$dao->gifts = [];

			return;
		}

		if (array_key_exists($dao->getId(), $this->loadedFor)) {
			$dao->gifts += $this->loadedFor[$dao->getId()];

			return;
		}

		$this->loadedFor[$dao->getId()] = [];

		$categories = array_merge([$dao->getDefaultCategory()], $dao->categories);
		$gifts      = $this->categoryGifts->findGifts($dao->price, $categories);

		if ($gifts)
			$dao->gifts += $gifts;

		$this->loadedFor[$dao->getId()] = $gifts;
	}

	public function fillDaoCartItem(FillDaoItemsEvent $event): void
	{
		if (!$this->cartsService->cDaoItems)
			return;

		$productIds = array_map(function($v) { return $v->productId; }, $this->cartsService->cDaoItems);
		$products   = $this->productsFacade->getProducts($productIds);

		/** @var Gift[][] $gifts */
		$gifts = [];
		foreach ($products as $product)
			if ($product->categoryGiftsAllowed)
				$gifts[$product->getId()] = $this->categoryGifts->findGifts($product->basePrice, array_merge($product->categories, [$product->getDefaultCategory()]));

		foreach ($this->cartsService->cDaoItems as $cartItem) {
			if (!isset($gifts[$cartItem->getProductId()]))
				continue;

			foreach ($gifts[$cartItem->getProductId()] as $gift)
				$cartItem->addGift(new CartItemGift($gift->getProduct(), $gift->getName()));
		}
	}
}
