<?php declare(strict_types = 1);

namespace EshopGifts\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopGifts\AdminModule\Model\Gifts;
use EshopGifts\Model\Entities\Gift;
use EshopGifts\Model\Entities\GiftText;
use Exception;
use Nette\Application\Attributes\Persistent;
use Nette\Application\ForbiddenRequestException;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;

class GiftForm extends BaseControl
{
	#[Persistent]
	public ?int     $giftId = null;
	protected ?Gift $gift   = null;

	public function __construct(
		protected Gifts        $giftsService,
		protected Products     $productsService,
		protected CacheService $cacheService,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Handler
	 */

	public function handleProductsLoad(): void
	{
		$output      = [];
		$conn        = $this->em->getConnection();
		$currentLang = $this->translator->getLocale();

		foreach ($conn->fetchAllAssociative("SELECT p.id, p.code1, p.ean, pt.name, pt.lang 
				FROM eshop_catalog__product p
				INNER JOIN eshop_catalog__product_texts pt ON pt.id = p.id AND pt.name IS NOT NULL
				WHERE p.is_deleted = 0") as $row) {
			/** @var array $row */
			if (!$row['name']) {
				continue;
			}

			if (!isset($output[$row['id']])) {
				$output[$row['id']] = [
					'id'    => $row['id'],
					'code1' => $row['code1'],
					'ean'   => $row['ean'],
					'texts' => [],
				];
			}

			$output[$row['id']]['texts'][$row['lang']] = $row['name'];

			if ($row['lang'] === $currentLang) {
				$output[$row['id']]['name'] = $row['name'];
			}
		}

		$this->presenter->sendJson($output);
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->getElementPrototype()->setAttribute('id', 'eshopGifts_categoryGiftForm');
		$form->addHidden('productId');

		$form->addText('product', 'eshopGifts.gift.product')
			->setHtmlAttribute('id', 'categoryGiftForm_product')
			->setHtmlAttribute('data-autocomplete-name', 'eshopGiftsGiftForm')
			->setHtmlAttribute('data-autocomplete-keys', 'id,code1,ean,name')
			->setHtmlAttribute('data-autocomplete-url', $this->link('productsLoad!'))
			->setRequired();
		$form->addText('name', 'eshopGifts.gift.name')
			->setMaxLength(255)
			->setNullable()
			->setIsMultilanguage();
		$form->addEditor('description', 'eshopGifts.gift.description')
			->setMaxLength(255)
			->setToolbar('Text')
			->setDisableAutoP(false)
			->setHeight(100)
			->setNullable()
			->setIsMultilanguage();
		$form->addBool('isActive', 'eshopGifts.gift.isActive')->setDefaultValue(1);

		$form->addSaveCancelControl();

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();

			/** @var GiftText[] $giftTexts */
			$giftTexts = [];
			$product   = $this->productsService->getReference($values->productId);

			if ($this->getGift() instanceof Gift) {
				$gift      = $this->getGift();
				$giftTexts = $gift->getTexts()->toArray();
			} else {
				$gift = new Gift($product);
			}

			foreach ($langValues as $l => $v) {
				if (Validators::isNone($v['name'])) {
					if (isset($giftTexts[$l])) {
						$this->em->remove($giftTexts[$l]);
						unset($giftTexts[$l]);
					}
					continue;
				}

				if (!isset($giftTexts[$l])) {
					$giftTexts[$l] = new GiftText($gift, $l);
				}
				$giftTexts[$l]->name        = $v['name'];
				$giftTexts[$l]->description = $v['description'];

				$this->em->persist($giftTexts[$l]);
			}

			$gift->setTexts($giftTexts);
			$gift->setProduct($product);
			$gift->setActive($values->isActive);

			$this->em->persist($gift);
			$this->em->flush();

			$this->presenter->flashMessageSuccess($this->t('eshopGifts.gift.saved'));

			$this->cacheService->defaultCache->remove('categoryGiftsAll');
			$this->cacheService->defaultCache->clean([
				Cache::Tags => 'gifts',
			]);
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ======================== GET / SET
	 */

	public function getGift(): ?Gift
	{
		if ($this->giftId && !$this->gift) {
			$this->gift = $this->giftsService->get($this->giftId);
		}

		return $this->gift;
	}

	public function setGift(int $id): void
	{
		$gift = $this->giftsService->get($id);

		if (!$gift) {
			throw new ForbiddenRequestException("Gift '$id' not found");
		}

		$this->gift   = $gift;
		$this->giftId = $id;

		$this['form']->setDefaults([
			'productId' => $gift->getProduct()->getId(),
			'product'   => $gift->getProduct()->code1 . ' | ' . $gift->getProduct()->getText()?->name,
			'isActive'  => $gift->isActive(),
		]);

		// texts
		$gTexts = [];
		foreach ($gift->getTexts()->toArray() as $lang => $texts) {
			foreach (['name', 'description'] as $v) {
				$gTexts[$v][$lang] = $texts->$v;
			}
		}
		$this['form']->setDefaults($gTexts);
	}

}
