<?php declare(strict_types = 1);

namespace EshopGifts\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Config;
use EshopGifts\AdminModule\Model\Gifts;
use EshopGifts\Model\Entities\Gift;
use EshopStock\DI\EshopStockExtension;
use Nette\Application\AbortException;
use Nette\Utils\Html;

class GiftsGrid extends BaseControl
{
	public function __construct(
		protected Gifts            $giftsService,
		protected IGiftFormFactory $giftFormFactory,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('eshopGifts.title.newGift');
		$this->template->modal      = 'giftForm';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this->template->modalTitle = $this->t('eshopGifts.title.editGift');
		$this['form']->setGift($id);
		$this->template->modal = 'giftForm';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$p = $this->presenter;
		if ($this->giftsService->remove($id)) {
			$p->flashMessageSuccess('default.removed');
			$this['grid']->reload();
		} else {
			$p->flashMessageDanger('default.removeFailed');
		}

		$p->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->giftsService->getEr()->createQueryBuilder('g')
			->addSelect('p, pt')
			->innerJoin('g.product', 'p')
			->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale());

		$grid->setDataSource($qb);

		// Columns
		//TODO obrázek
		$grid->addColumnText('product', 'eshopGifts.gift.product')->setRenderer(function(Gift $row) {
			$product = $row->getProduct();

			return Html::el('a', [
				'href' => $this->presenter->link(':EshopCatalog:Admin:Products:edit', [$product->getId()]),
			])->setText($product->getText()?->name ?: '');
		});
		$grid->addColumnText('code1', 'eshopGifts.gift.code1', 'product.code1');
		$grid->addColumnText('ean', 'eshopGifts.gift.ean', 'product.ean');

		if (Config::load('pseudoWarehouse') || class_exists(EshopStockExtension::class)) {
			$grid->addColumnText('quantity', 'eshopGifts.gift.quantity')->setRenderer(function(Gift $gift) {
				$html = Html::el('span', ['class' => 'btn btn-xs']);
				if ($gift->getProduct()->quantity > 0) {
					$html->addClass('btn-success')->setText($gift->getProduct()->quantity);
				} else if ($gift->getProduct()->unlimitedQuantity) {
					$html->addClass('btn-success')->setText('∞');
				} else {
					$html->addClass('btn-danger')->setText($gift->getProduct()->quantity);
				}

				return $html;
			})->setAlign('right');
		}

		$grid->addColumnStatus('isActive', 'default.isActive')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = $this->gridPublishChange(...);

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->addClass('ajax')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm('default.reallyDelete');

		// Prototype
		$grid->getColumn('code1')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('ean')->getElementPrototype('td')->addClass('w1nw');
		if (Config::load('pseudoWarehouse') || class_exists(EshopStockExtension::class)) {
			$grid->getColumn('quantity')->getElementPrototype('td')->addClass('w1nw');
		}

		return $grid;
	}

	protected function createComponentForm(): GiftForm
	{
		$control = $this->giftFormFactory->create();

		$control->onAnchor[] = function() use ($control): void {
			$control['form']->onSuccessSave[] = function(): void {
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->onSuccessSaveAndClose[] = function(): void {
				$this->presenter->payload->hideModal = true;
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->getComponent('saveControl')->closeModalOnCancel();
		};

		return $control;
	}

	/**
	 * @param int      $id
	 * @param int|bool $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->giftsService->setActive($id, $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

}
