<?php declare(strict_types = 1);

namespace EshopGifts\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use EshopCatalog\AdminModule\Components\Products\ProductForm;
use EshopCatalog\Model\Entities\Product;
use EshopGifts\AdminModule\Model\Gifts;
use EshopGifts\AdminModule\Model\ProductGifts;
use EshopGifts\Model\Entities\Gift;
use EshopGifts\Model\Entities\ProductGift;
use Nette\Utils\ArrayHash;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ProductFormSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected Gifts                  $giftsService,
		protected ProductGifts           $productGiftsService,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			ProductForm::class . '::createForm'  => 'createForm',
			ProductForm::class . '::formSuccess' => 'formSuccess',
			ProductForm::class . '::setProduct'  => 'setProduct',
		];
	}

	public function createForm(CreateFormEvent $event): void
	{
		$container = $event->form->addContainer('gifts');
		$container->addCustomData('template', __DIR__ . '/ProductFormGifts.latte');

		foreach ($this->giftsService->getAll() as $gift) {
			$giftContainer = $container->addContainer('g' . $gift->getId());
			$giftContainer->addHidden('id', (string) $gift->getId());
			$giftContainer->addBool('isActive', $gift->getText()?->name ?: '')
				->setTranslator(null)
				->setDefaultValue(0);
			$giftContainer->addDatePicker('dateFrom', 'eshopGifts.gift.dateFrom')
				->setNullable();
			$giftContainer->addDatePicker('dateTo', 'eshopGifts.gift.dateTo')
				->setNullable();
		}
	}

	public function formSuccess(FormSuccessEvent $event): void
	{
		/** @var Product $product */
		$product = $event->custom['entity'];
		$values  = $event->values;

		if (!$product->getId()) {
			return;
		}

		$currentGifts = $this->productGiftsService->getByProduct($product->getId());

		foreach ($values->gifts as $v) {
			/** @var ArrayHash $v */
			if ($v->isActive === 1) {
				/** @var Gift $giftEntity */
				$giftEntity   = $this->giftsService->getReference($v->id);
				$pg           = $currentGifts[$v->id] ?? new ProductGift($giftEntity, $product);
				$pg->dateFrom = $v->dateFrom;
				$pg->dateTo   = $v->dateTo;
				$this->em->persist($pg);
			} else if (isset($currentGifts[$v->id])) {
				$this->em->remove($currentGifts[$v->id]);
			}
		}
	}

	public function setProduct(SetFormDataEvent $event): void
	{
		/** @var Product $product */
		$product = $event->entity;

		if (!$product->getId()) {
			return;
		}

		$gifts = $this->productGiftsService->getByProduct($product->getId());

		$d = [];
		foreach ($gifts as $gift) {
			$d['g' . $gift->getGift()->getId()] = [
				'isActive' => 1,
				'dateFrom' => $gift->dateFrom,
				'dateTo'   => $gift->dateTo,
			];
		}

		if ($d !== []) {
			$event->form->getComponent('gifts')->setDefaults($d);
		}
	}
}
