<?php declare(strict_types = 1);

namespace EshopGifts\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\Query\Parameter;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopGifts\Model\Entities\CategoryGift;
use Nette\Application\UI\InvalidLinkException;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;
use Throwable;

class CategoryGifts extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = CategoryGift::class;

	/** @var array|null */
	protected ?array $cGifts = null;

	public function __construct(
		protected Gifts        $giftsService,
		protected CacheService $cacheService,
	)
	{
	}

	/**
	 *
	 * @return Dao\Gift[]
	 * @throws NonUniqueResultException
	 * @throws InvalidLinkException
	 * @throws Throwable
	 */
	public function findGifts(float $price, array $categories): array
	{
		$giftIds = [];
		$result  = [];

		foreach ($categories as $k => $v) {
			if ($v instanceof Category) {
				$categories[$k] = $v->getId();
			} else if (!is_numeric($v)) {
				unset($categories[$k]);
			}
		}

		if ($categories === []) {
			return [];
		}

		foreach ($this->getAll() as $row) {
			if (($price >= $row['priceFrom'])
				&& ($price <= $row['priceTo'] || $row['priceTo'] === null)
				&& (array_intersect($categories, $row['categories']))
			) {
				$giftIds[] = $row['gift'];
			}
		}

		foreach ($giftIds !== [] ? $this->giftsService->getByIds($giftIds) : [] as $gift) {
			$gift->addOrigin        = 'category';
			$result[$gift->getId()] = $gift;
		}

		return $result;
	}

	public function getAll(): array
	{
		if ($this->cGifts === null) {
			$today = (new DateTime())->setTime(0, 0, 0);

			$this->cGifts = (array) $this->cacheService->defaultCache->load('categoryGiftsAll', function(&$dep) use ($today) {
				$dep = [
					Cache::Tags   => [Categories::CACHE_NAMESPACE, 'gifts'],
					Cache::Expire => '1 week',
				];

				$arr = [];
				foreach ($this->getEr()->createQueryBuilder('cg')
					         ->addSelect('GROUP_CONCAT(cats.id) as categories, IDENTITY(cg.gift) as gift')
					         ->innerJoin('cg.categories', 'cats')
					         ->where('cg.dateFrom <= :today OR cg.dateFrom IS NULL')
					         ->andWhere('cg.dateTo >= :today OR cg.dateTo IS NULL')
					         ->andWhere('cg.isActive = 1')
					         ->setParameters(new ArrayCollection([new Parameter('today', $today)]))
					         ->groupBy('cg.id')
					         ->getQuery()->getArrayResult() as $oRow) {
					/** @var array $oRow */
					$row = $oRow[0];
					$id  = (int) $oRow['gift'];

					$cats = [];
					foreach (explode(',', (string) $oRow['categories']) as $k => $v) {
						if (is_numeric($v)) {
							$cats[] = (int) $v;
						}
					}

					if ($cats === []) {
						continue;
					}

					$arr[] = [
						'gift'       => $id,
						'priceFrom'  => $row['priceFrom'] ? (float) $row['priceFrom'] : null,
						'priceTo'    => $row['priceTo'] ? (float) $row['priceTo'] : null,
						'categories' => $cats,
					];
				}

				return $arr;
			});
		}

		return $this->cGifts;
	}
}
