<?php declare(strict_types = 1);

namespace EshopGifts\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Entities\Category;
use EshopGifts\AdminModule\Model\CategoryGifts;
use EshopGifts\Model\Entities\CategoryGift;
use Nette\Application\AbortException;

class CategoryGiftsGrid extends BaseControl
{
	protected CategoryGifts            $categoryGifts;
	protected ICategoryGiftFormFactory $formFactory;

	public function __construct(
		ICategoryGiftFormFactory $formFactory,
		CategoryGifts            $categoryGifts
	)
	{
		$this->formFactory   = $formFactory;
		$this->categoryGifts = $categoryGifts;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle       = $this->t('eshopGifts.title.newCategoryGift');
		$this->template->modal            = 'categoryGiftForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this->template->modalTitle = $this->t('eshopGifts.title.editCategoryGift');
		$this['form']->setCategoryGift($id);
		$this->template->modal            = 'categoryGiftForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$p = $this->presenter;
		if ($this->categoryGifts->remove($id)) {
			$p->flashMessageSuccess('default.removed');
			$this['grid']->reload();
		} else {
			$p->flashMessageDanger('default.removeFailed');
		}

		$p->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->categoryGifts->getEr()->createQueryBuilder('cg')
			->addSelect('gift, gt, product, cats, catsT')
			->innerJoin('cg.gift', 'gift')
			->innerJoin('gift.texts', 'gt', Join::WITH, 'gt.lang = :lang')
			->innerJoin('gift.product', 'product')
			->leftJoin('cg.categories', 'cats')
			->leftJoin('cats.categoryTexts', 'catsT', Join::WITH, 'catsT.lang = :lang')
			->setParameter('lang', $this->translator->getLocale());

		$grid->setDataSource($qb);

		// Columns
		//TODO obrázek
		$grid->addColumnText('gift', 'eshopGifts.gift.gift', 'gift.text.name');
		$grid->addColumnText('categories', 'eshopGifts.gift.categories')
			->setRenderer(fn(CategoryGift $row) => implode(', ', array_map(static fn(Category $v) => $v->getCategoryText()->getName(), $row->getCategories()->toArray())));
		$grid->addColumnPrice('priceFrom', 'eshopGifts.gift.priceFrom');
		$grid->addColumnPrice('priceTo', 'eshopGifts.gift.priceTo');
		$grid->addColumnDateTime('dateFrom', 'eshopGifts.gift.dateFrom')->setFormat('j. n. Y');
		$grid->addColumnDateTime('dateTo', 'eshopGifts.gift.dateTo')->setFormat('j. n. Y');
		$grid->addColumnStatus('isActive', 'default.isActive')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->addClass('ajax')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	protected function createComponentForm(): CategoryGiftForm
	{
		$control = $this->formFactory->create();

		if ($this->getParameter('id')) {
			$control->setCategoryGift((int) $this->getParameter('id'));
		}

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[] = function() {
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->onSuccessSaveAndClose[] = function() {
				$this->presenter->payload->hideModal = true;
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->getComponent('saveControl')->closeModalOnCancel();
		};

		return $control;
	}

	/**
	 * @param int      $id
	 * @param int|bool $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->categoryGifts->setActive($id, $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}
}
