<?php declare(strict_types = 1);

namespace EshopGifts\AdminModule\Components;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopGifts\AdminModule\Model\Gifts;
use EshopGifts\AdminModule\Model\OrderGifts;
use EshopGifts\Model\Entities\OrderGift;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use Exception;
use Nette\Application\ForbiddenRequestException;
use Nette\Utils\ArrayHash;

class OrderGiftForm extends BaseControl
{
	/** @var int @persistent */
	public                    $orderGiftId;
	protected ?OrderGift      $orderGift = null;
	protected OrderGifts      $orderGifts;
	protected Gifts           $giftsService;
	protected Sites           $sitesService;
	protected GroupsCustomers $groupsCustomers;

	public function __construct(
		OrderGifts      $orderGifts,
		Gifts           $giftsService,
		Sites           $sitesService,
		GroupsCustomers $groupsCustomers
	)
	{
		$this->orderGifts      = $orderGifts;
		$this->giftsService    = $giftsService;
		$this->sitesService    = $sitesService;
		$this->groupsCustomers = $groupsCustomers;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$sites          = $this->sitesService->getOptionsForSelect();
		$customerGroups = $this->groupsCustomers->getOptionsForSelect();

		$gifts = [];
		foreach ($this->giftsService->getEr()->createQueryBuilder('g')->select('g.id, gt.name, p.code1, p.ean')
			         ->innerJoin('g.product', 'p')
			         ->innerJoin('g.texts', 'gt', Join::WITH, 'gt.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->getQuery()->getArrayResult() as $row) {
			$gifts[$row['id']] = $row['name'] . ' (' . implode(', ', [$row['code1'], $row['ean']]) . ')';
		}

		$form->addSelect('gift', 'eshopGifts.gift.gift', $gifts);
		$form->addCheckboxList('sites', 'eshopGifts.gift.sites', $sites);
		$form->addCheckbox('withoutCustomerGroup', 'eshopGifts.gift.withoutCustomerGroup');
		$form->addCheckboxList('customerGroups', 'eshopGifts.gift.customerGroups', $customerGroups);
		$form->addBool('isActive', 'eshopGifts.gift.isActive')->setDefaultValue(1);
		$form->addText('priceFrom', 'eshopGifts.gift.orderPriceFrom')
			->setHtmlAttribute('min', 0)
			->setNullable()
			->addCondition($form::FILLED)
			->addRule($form::FLOAT);
		$form->addText('priceTo', 'eshopGifts.gift.orderPriceTo')
			->setHtmlAttribute('min', 0)
			->setNullable()
			->addCondition($form::FILLED)
			->addRule($form::FLOAT);

		$form->addDatePicker('dateFrom', 'eshopGifts.gift.dateFrom');
		$form->addDatePicker('dateTo', 'eshopGifts.gift.dateTo');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$gift = $this->giftsService->getReference($values->gift);

			$orderGift = $this->getOrderGift() ?: new OrderGift($gift);
			$orderGift->setGift($gift);
			$orderGift->setActive($values->isActive);
			$orderGift->priceFrom            = $values->priceFrom;
			$orderGift->priceTo              = $values->priceTo;
			$orderGift->dateFrom             = $values->dateFrom;
			$orderGift->dateTo               = $values->dateTo;
			$orderGift->withoutCustomerGroup = (int) $values->withoutCustomerGroup;

			$sites = [];
			foreach ($values->sites as $v) {
				$sites[$v] = $this->sitesService->getReference($v);
			}
			$orderGift->setSites($sites);

			$customerGroups = [];
			foreach ($values->customerGroups as $v) {
				$customerGroups[$v] = $this->groupsCustomers->getReference($v);
			}
			$orderGift->setCustomerGroups($customerGroups);

			$this->em->persist($orderGift)->flush();
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ======================== GET / SET
	 */

	public function getOrderGift(): ?OrderGift
	{
		if ($this->orderGiftId && !$this->orderGift) {
			$this->orderGift = $this->orderGifts->get($this->orderGiftId);
		}

		return $this->orderGift;
	}

	public function setOrderGift(int $id): void
	{
		$orderGift = $this->orderGifts->get($id);

		if (!$orderGift) {
			throw new ForbiddenRequestException("Order gift '$id' not found");
		}

		$this->orderGift   = $orderGift;
		$this->orderGiftId = $id;

		$this['form']->setDefaults([
			'gift'                 => $orderGift->getGift()->getId(),
			'sites'                => array_keys($orderGift->getSites()->toArray()),
			'customerGroups'       => array_keys($orderGift->getCustomerGroups()->toArray()),
			'isActive'             => $orderGift->isActive(),
			'priceFrom'            => $orderGift->priceFrom,
			'priceTo'              => $orderGift->priceTo,
			'dateFrom'             => $orderGift->dateFrom,
			'dateTo'               => $orderGift->dateTo,
			'withoutCustomerGroup' => $orderGift->withoutCustomerGroup,
		]);
	}

}
