<?php declare(strict_types = 1);

namespace EshopGifts\FrontModule\Model;

use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use Core\Model\Helpers\BaseFrontEntityService;
use EshopGifts\FrontModule\Model\Dao\Gift;
use EshopGifts\Model\Entities\CategoryGift;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;
use EshopGifts\FrontModule\Model\Dao;

class CategoryGifts extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = CategoryGift::class;

	/** @var Gift[] */
	protected $cGifts;

	protected Gifts        $giftsService;
	protected CacheService $cacheService;

	public function __construct(
		Gifts        $gifts,
		CacheService $cacheService
	)
	{
		$this->giftsService = $gifts;
		$this->cacheService = $cacheService;
	}

	/**
	 * @param float $price
	 * @param array $categories
	 *
	 * @return Dao\Gift[]
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Nette\Application\UI\InvalidLinkException
	 * @throws \Throwable
	 */
	public function findGifts(float $price, array $categories): array
	{
		$giftIds = [];
		$result  = [];

		foreach ($categories as $k => $v) {
			if ($v == null)
				unset($categories[$k]);
			else if ($v instanceof Category)
				$categories[$k] = $v->getId();
		}

		foreach ($this->getAll() as $row) {
			if (($price >= $row['priceFrom'] || $row['priceFrom'] === null)
				&& ($price <= $row['priceTo'] || $row['priceTo'] === null)
				&& (empty($categories) || array_intersect($categories, $row['categories']))
			) {
				$giftIds[] = $row['gift'];
			}
		}

		foreach ($giftIds ? $this->giftsService->getByIds($giftIds) : [] as $gift) {
			$gift->addOrigin        = 'category';
			$result[$gift->getId()] = $gift;
		}

		return $result;
	}

	public function getAll(): array
	{
		if ($this->cGifts === null) {
			$today = (new DateTime())->setTime(0, 0, 0);

			$this->cGifts = $this->cacheService->defaultCache->load('categoryGiftsAll', function(&$dep) use ($today) {
				$dep = [
					Cache::Tags       => [Categories::CACHE_NAMESPACE, 'gifts'],
					Cache::EXPIRATION => '1 week',
				];

				$arr = [];
				foreach ($this->getEr()->createQueryBuilder('cg')
					         ->addSelect('GROUP_CONCAT(cats.id) as categories, IDENTITY(cg.gift) as gift')
					         ->innerJoin('cg.categories', 'cats')
					         ->where('cg.dateFrom <= :today OR cg.dateFrom IS NULL')
					         ->andWhere('cg.dateTo >= :today OR cg.dateTo IS NULL')
					         ->andWhere('cg.isActive = 1')
					         ->setParameters([
						         'today' => $today,
					         ])
					         ->groupBy('cg.id')
					         ->getQuery()->getArrayResult() as $oRow) {
					$row   = $oRow[0];
					$id    = (int) $oRow['gift'];
					$arr[] = [
						'gift'       => $id,
						'priceFrom'  => $row['priceFrom'] ? (float) $row['priceFrom'] : null,
						'priceTo'    => $row['priceTo'] ? (float) $row['priceTo'] : null,
						'categories' => explode(',', $oRow['categories']),
					];
				}

				return $arr;
			});
		}

		return $this->cGifts;
	}
}
