<?php declare(strict_types = 1);

namespace EshopGifts\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopGifts\Model\Entities\Gift;
use EshopGifts\FrontModule\Model\Dao;
use Throwable;

class Gifts extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = Gift::class;
	/** @var Dao\Gift[]|null */
	protected ?array         $cGifts = null;
	protected ?array         $cAll   = null;
	protected ProductsFacade $productsFacade;

	public function __construct(ProductsFacade $productsFacade)
	{
		$this->productsFacade = $productsFacade;
	}

	/**
	 * @return array<int, Dao\Gift>
	 * @throws Throwable
	 */
	public function getAll(): array
	{
		if ($this->cAll === null) {
			$this->cAll = [];

			foreach ($this->getEr()->createQueryBuilder('g')
				         ->select('IDENTITY(g.product) as product, g.id, gt.name, gt.description')
				         ->innerJoin('g.texts', 'gt', Join::WITH, 'gt.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->where('g.isActive = 1')
				         ->getQuery()->getScalarResult() as $row) {
				$this->cAll[$row['id']] = $row;
			}
		}

		return $this->cAll;
	}

	/**
	 * @param int[]|string[] $ids
	 *
	 * @return Dao\Gift[]
	 * @throws Throwable
	 */
	public function getByIds(array $ids): array
	{
		$result   = [];
		$gifts    = $this->getAll();
		$whereIds = [];

		foreach ($ids as $id) {
			if (isset($gifts[$id])) {
				if (isset($this->cGifts[$id])) {
					$result[$id] = $this->cGifts[$id];
				} else {
					$whereIds[] = $id;
				}
			}
		}

		if (!empty($whereIds)) {
			$productsIds = array_map(static fn($v) => $gifts[$v]['product'], $whereIds);
			$products    = $this->productsFacade->getProducts($productsIds);

			foreach ($whereIds as $giftId) {
				$rawGift = $gifts[$giftId];

				/** @var Product|null $product */
				$product = $products[$rawGift['product']];

				if (!$product || !$product->canAddToCart) {
					continue;
				}

				$gift        = new Dao\Gift((int) $rawGift['id'], $product, $rawGift['name'], $rawGift['description']);
				$gift->code1 = $product->getCode1();
				$gift->ean   = $product->getEan();

				$this->cGifts[$rawGift['id']] = $gift;
				$result[$rawGift['id']]       = $gift;
			}
		}

		return $result;
	}

}
