<?php declare(strict_types = 1);

namespace EshopGifts\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Sites;
use Currency\Model\Exchange;
use Doctrine\ORM\Query\Expr\Join;
use EshopGifts\FrontModule\Model\Dao\Gift;
use EshopGifts\Model\Entities\OrderGift;
use EshopOrders\FrontModule\Model\Customers;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;
use Nette\Utils\Validators;
use Users\Model\Security\User;

class OrderGifts extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = OrderGift::class;

	/** @var Gift[] */
	protected $cGifts;

	/** @var Gifts */
	protected $giftsService;

	/** @var Sites */
	protected $sitesService;

	protected User      $user;
	protected Customers $customers;
	protected Exchange  $exchange;

	public function __construct(
		Gifts     $gifts,
		Sites     $sites,
		User      $user,
		Customers $customers,
		Exchange  $exchange
	)
	{
		$this->giftsService = $gifts;
		$this->sitesService = $sites;
		$this->user         = $user;
		$this->customers    = $customers;
		$this->exchange     = $exchange;
	}

	/**
	 * @param float $price
	 *
	 * @return Dao\Gift[]
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Nette\Application\UI\InvalidLinkException
	 * @throws \Throwable
	 */
	public function findGifts(float $price): array
	{
		$giftIds = [];

		foreach ($this->getAll() as $row) {
			if (($price >= $row['priceFrom'] || $row['priceFrom'] === null)
				&& ($price <= $row['priceTo'] || $row['priceTo'] === null)
			) {
				$giftIds[] = $row['gift'];
			}
		}

		return $giftIds ? $this->giftsService->getByIds($giftIds) : [];
	}

	/**
	 * @return Dao\Gift[]
	 */
	public function findAll(): array
	{
		$giftIds = [];
		$all     = $this->getAll();

		foreach ($all as $row) {
			$giftIds[] = $row['gift'];
		}

		$result = $giftIds ? $this->giftsService->getByIds($giftIds) : [];
		foreach ($result as $k => $row) {
			if (isset($all[$k])) {
				$row->priceFrom = $all[$k]['priceFrom'];
			}
		}

		return $result;
	}

	public function getAll(): array
	{
		if ($this->cGifts === null) {
			$this->cGifts = [];
			$today        = (new DateTime)->setTime(0, 0);
			$customer     = $this->user->isLoggedIn() ? $this->customers->getByUserId($this->user->getId()) : null;

			$qb = $this->getEr()->createQueryBuilder('og')
				->addSelect('IDENTITY(og.gift) as gift')
				->where('og.dateFrom <= :today OR og.dateFrom IS NULL')
				->andWhere('og.dateTo >= :today OR og.dateTo IS NULL')
				->andWhere('og.isActive = 1')
				->innerJoin('og.sites', 'site', Join::WITH, 'site.ident = :site')
				->setParameter('today', $today)
				->setParameter('site', $this->sitesService->getCurrentSite()->getIdent());

			$qb2 = $this->getEr()->createQueryBuilder('og2');
			$qb2->select('GROUP_CONCAT(custGroups.id)')
				->join('og2.customerGroups', 'custGroups')
				->where('og2.id = og.id');

			$qb->addSelect('(' . $qb2->getDQL() . ') as customerGroupIds')
				->leftJoin('og.customerGroups', 'customerGroups');

			foreach ($qb->getQuery()->getArrayResult() as $oRow) {
				$customerGroupIds = $oRow['customerGroupIds'];
				$allowByGroup     = true;
				$allowByNoGroup   = true;
				if (!Validators::isNone($customerGroupIds)) {
					if (Strings::contains($customerGroupIds, ',')) {
						$groupIds = array_map(static fn($id) => (int) trim($id), explode(',', $customerGroupIds));
					} else {
						$groupIds = [(int) trim($customerGroupIds)];
					}

					if ($customer === null || !$customer->getGroupCustomers() || !in_array($customer->getGroupCustomers()->getId(), $groupIds)) {
						$allowByGroup = false;
					}
				} else {
					$allowByGroup = false;
				}

				$row = $oRow[0];

				if ((bool) $oRow[0]['withoutCustomerGroup'] && $customer && $customer->getGroupCustomers()) {
					$allowByNoGroup = false;
				}

				if (!$allowByGroup && !$allowByNoGroup) {
					continue;
				}

				$id = (int) $oRow['gift'];

				$priceFrom = $row['priceFrom'] ? (float) $row['priceFrom'] : null;
				if ($priceFrom) {
					$priceFrom = round($this->exchange->change($priceFrom), 2);
				}

				$priceTo = $row['priceTo'] ? (float) $row['priceTo'] : null;
				if ($priceTo) {
					$priceTo = round($this->exchange->change($priceTo), 2);
				}

				$this->cGifts[$id] = [
					'gift'      => $id,
					'priceFrom' => $priceFrom,
					'priceTo'   => $priceTo,
				];
			}
		}

		return $this->cGifts;
	}
}
