<?php declare(strict_types = 1);

namespace EshopGifts\AdminModule\Components;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\Category;
use EshopGifts\AdminModule\Model\CategoryGifts;
use EshopGifts\AdminModule\Model\Gifts;
use EshopGifts\Model\Entities\CategoryGift;
use EshopGifts\Model\Entities\Gift;
use Exception;
use Nette\Application\Attributes\Persistent;
use Nette\Application\ForbiddenRequestException;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;

class CategoryGiftForm extends BaseControl
{
	#[Persistent]
	public ?int             $categoryGiftId = null;
	protected ?CategoryGift $categoryGift   = null;

	public function __construct(
		protected CategoryGifts $categoryGifts,
		protected Gifts         $giftsService,
		protected Categories    $categoriesService,
		protected Sites         $sitesService,
		protected CacheService  $cacheService,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$sites = $this->sitesService->getAll();
		if (count($sites) > 1) {
			$flat = [];
			foreach ($sites as $site) {
				$tmp = $this->categoriesService->getFlatTree($site->getIdent());

				if ($tmp === []) {
					continue;
				}

				$flat[] = [
					'id'     => $tmp[0]['parent'],
					'parent' => 0,
					'name'   => $site->getIdent(),
				];

				$flat = array_merge($flat, $tmp);
			}
		} else {
			$flat = $this->categoriesService->getFlatTree();
		}

		$gifts = [];
		foreach ($this->giftsService->getEr()->createQueryBuilder('g')->select('g.id, gt.name')
			         ->innerJoin('g.texts', 'gt', Join::WITH, 'gt.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->getQuery()->getArrayResult() as $row) {
			/** @var array $row */
			$gifts[$row['id']] = $row['name'];
		}

		$form->addSelect('gift', 'eshopGifts.gift.gift', $gifts);
		$form->addBool('isActive', 'eshopGifts.gift.isActive')->setDefaultValue(1);
		$form->addCheckboxNestedList('categories', 'eshopGifts.gift.categories', $flat);
		$form->addText('priceFrom', 'eshopGifts.gift.priceFrom')
			->setHtmlAttribute('min', 0)
			->setNullable()
			->addCondition($form::FILLED)
			->addRule($form::FLOAT);
		$form->addText('priceTo', 'eshopGifts.gift.priceTo')
			->setHtmlAttribute('min', 0)
			->setNullable()
			->addCondition($form::FILLED)
			->addRule($form::FLOAT);

		$form->addDatePicker('dateFrom', 'eshopGifts.gift.dateFrom');
		$form->addDatePicker('dateTo', 'eshopGifts.gift.dateTo');

		$form->addSaveCancelControl();

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			/** @var Gift $gift */
			$gift = $this->giftsService->getReference((int) $values->gift);

			$categories = [];
			foreach ($values->categories as $v) {
				if (!$v) {
					continue;
				}

				/** @var Category $category */
				$category     = $this->categoriesService->getReference((int) $v);
				$categories[] = $category;
			}

			$categoryGift = $this->getCategoryGift() ?? new CategoryGift($gift);
			$categoryGift->setGift($gift);
			$categoryGift->setCategories($categories);
			$categoryGift->setActive($values->isActive);
			$categoryGift->priceFrom = $values->priceFrom;
			$categoryGift->priceTo   = $values->priceTo;
			$categoryGift->dateFrom  = $values->dateFrom;
			$categoryGift->dateTo    = $values->dateTo;

			$this->em->persist($categoryGift);
$this->em->flush();

			$this->presenter->flashMessageSuccess($this->t('eshopGifts.gift.saved'));

			$this->cacheService->defaultCache->remove('categoryGiftsAll');
			$this->cacheService->defaultCache->clean([
				Cache::Tags => 'gifts',
			]);
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ======================== GET / SET
	 */

	public function getCategoryGift(): ?CategoryGift
	{
		if ($this->categoryGiftId && !$this->categoryGift) {
			$this->categoryGift = $this->categoryGifts->get($this->categoryGiftId);
		}

		return $this->categoryGift;
	}

	public function setCategoryGift(int $id): void
	{
		$categoryGift = $this->categoryGifts->get($id);

		if (!$categoryGift instanceof CategoryGift) {
			throw new ForbiddenRequestException("Category gift '$id' not found");
		}

		$this->categoryGift   = $categoryGift;
		$this->categoryGiftId = $id;

		$categories = [];
		foreach ($categoryGift->getCategories() as $cat) {
			if ($cat->getId() && array_key_exists($cat->getId(), $this['form']->getComponent('categories')->getItems())) {
				$categories[] = $cat->getId();
			}
		}

		$this['form']->setDefaults([
			'gift'       => $categoryGift->getGift()->getId(),
			'categories' => $categories,
			'isActive'   => $categoryGift->isActive(),
			'priceFrom'  => $categoryGift->priceFrom,
			'priceTo'    => $categoryGift->priceTo,
			'dateFrom'   => $categoryGift->dateFrom,
			'dateTo'     => $categoryGift->dateTo,
		]);
	}
}
