<?php declare(strict_types = 1);

namespace EshopGifts\AdminModule\Components;

use Core\Model\Entities\Site;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\ORM\Query\Expr\Join;
use EshopGifts\AdminModule\Model\OrderGifts;
use EshopGifts\Model\Entities\OrderGift;
use Nette\Application\AbortException;

class OrderGiftsGrid extends BaseControl
{
	public function __construct(
		protected IOrderGiftFormFactory $formFactory,
		protected OrderGifts            $orderGifts,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('eshopGifts.title.newOrderGift');
		$this->template->modal      = 'orderGiftForm';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this->template->modalTitle = $this->t('eshopGifts.title.editOrderGift');
		$this['form']->setOrderGift($id);
		$this->template->modal = 'orderGiftForm';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$p = $this->presenter;
		if ($this->orderGifts->remove($id)) {
			$p->flashMessageSuccess('default.removed');
			$this['grid']->reload();
		} else {
			$p->flashMessageDanger('default.removeFailed');
		}

		$p->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->orderGifts->getEr()->createQueryBuilder('og')
			->addSelect('gift, gt, product, sites')
			->innerJoin('og.gift', 'gift')
			->innerJoin('gift.texts', 'gt', Join::WITH, 'gt.lang = :lang')
			->innerJoin('gift.product', 'product')
			->leftJoin('og.sites', 'sites')
			->setParameter('lang', $this->translator->getLocale());

		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnText('gift', 'eshopGifts.gift.gift', 'gift.text.name');
		$grid->addColumnText('sites', 'eshopGifts.gift.sites')->setRenderer(fn(OrderGift $row): string => implode(', ', array_map(static fn(Site $v): string => $v->getIdent(), $row->getSites()->toArray())));
		$grid->addColumnPrice('priceFrom', 'eshopGifts.gift.priceFrom');
		$grid->addColumnPrice('priceTo', 'eshopGifts.gift.priceTo');
		$grid->addColumnDateTime('dateFrom', 'eshopGifts.gift.dateFrom')->setFormat('j. n. Y');
		$grid->addColumnDateTime('dateTo', 'eshopGifts.gift.dateTo')->setFormat('j. n. Y');
		$grid->addColumnStatus('isActive', 'default.isActive')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = $this->gridPublishChange(...);

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->addClass('ajax')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	protected function createComponentForm(): OrderGiftForm
	{
		$control = $this->formFactory->create();
		$paramId = $this->getParameters()['id'] ?? null;

		if ($paramId) {
			$control->setOrderGift((int) $paramId);
		}

		$control->onAnchor[] = function() use ($control): void {
			$control['form']->onSuccessSave[] = function(): void {
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->onSuccessSaveAndClose[] = function(): void {
				$this->presenter->payload->hideModal = true;
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->getComponent('saveControl')->closeModalOnCancel();
		};

		return $control;
	}

	/**
	 * @param int      $id
	 * @param int|bool $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->orderGifts->setActive($id, $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}
}
