<?php declare(strict_types = 1);

namespace EshopGifts\FrontModule\Model\Subscribers;

use Core\Model\Event\DaoEvent;
use Core\Model\Module;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Event\ProductsEvent;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopGifts\FrontModule\Model\CategoryGifts;
use EshopGifts\FrontModule\Model\Dao\Gift;
use EshopGifts\FrontModule\Model\Gifts;
use EshopGifts\FrontModule\Model\OrderGifts;
use EshopGifts\FrontModule\Model\ProductGifts;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\FrontModule\Model\Dao\CartItemGift;
use EshopOrders\FrontModule\Model\Event\FillDaoEvent;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Throwable;

class SetGiftsToDaoSubscriber implements EventSubscriberInterface
{
	protected array $loadedFor = [];

	public function __construct(
		protected CategoryGifts  $categoryGifts,
		protected OrderGifts     $orderGifts,
		protected ProductGifts   $productGifts,
		protected ProductsFacade $productsFacade,
		protected Carts          $cartsService,
		protected Gifts          $gifts,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			Products::class . '::fillDao'      => 'fillDaoProduct',
			Products::class . '::afterFillDao' => 'afterFillDaoProduct',
			'eshopOrders.cartFillDao'          => ['cartFillDao', 90],
			'eshopOrders.cartFillDaoItems'     => ['fillDaoCartItem', 90],
		];
	}

	/**
	 * @return Gift[]
	 * @throws Throwable
	 */
	protected function findGifs(Product $product): array
	{
		if (isset($this->loadedFor[$product->getId()])) {
			return $this->loadedFor[$product->getId()];
		}

		$this->loadedFor[$product->getId()] = [];

		$gifts = $this->findCategoryGifts($product);

		if ($product->orderGiftsAllowed) {
			$gifts += $this->productGifts->findGifts($product->getId());
		}

		$this->loadedFor[$product->getId()] = $gifts;

		return $gifts;
	}

	/**
	 * @return Gift[]
	 * @throws Throwable
	 */
	protected function findCategoryGifts(Product $product): array
	{
		$categories = array_merge([$product->defaultCategoryId], $product->categories);

		return $product->categoryGiftsAllowed ? $this->categoryGifts->findGifts($product->getPrice(), $categories) : [];
	}

	public function fillDaoProduct(DaoEvent $event): void
	{
		/** @var Product $product */
		$product = &$event->dao;
		$gifts   = $this->findGifs($product);

		if ($gifts !== []) {
			$product->giftsIds = array_keys($gifts);
		}
	}

	public function afterFillDaoProduct(ProductsEvent $event): void
	{
		$products = &$event->products;

		foreach ($products as &$product) {
			if ($product->giftsIds === []) {
				$product->giftsIds += array_keys($this->findCategoryGifts($product));
			}

			if ($product->giftsIds) {
				$product->gifts += $this->gifts->getByIds($product->giftsIds);
			}
		}
	}

	public function fillDaoCartItem(FillDaoItemsEvent $event): void
	{
		if (!$this->cartsService->cDaoItems) {
			return;
		}

		$productIds = array_map(static fn(CartItem $v): ?int => $v->productId, $this->cartsService->cDaoItems);
		$productIds = array_filter($productIds, static fn(?int $v): bool => $v !== null);
		$products   = $this->productsFacade->getProducts($productIds);

		/** @var Gift[][] $gifts */
		$gifts = [];
		foreach ($products as $product) {
			foreach ($product->gifts as $v) {
				if (!$product->categoryGiftsAllowed && $v->addOrigin === 'category') {
					continue;
				}

				$gifts[$product->getId()][$v->getId()] = $v;
			}
		}

		foreach ($this->cartsService->cDaoItems as $cartItem) {
			if (!isset($gifts[$cartItem->getProductId()])) {
				continue;
			}

			foreach ($gifts[$cartItem->getProductId()] as $gift) {
				$cartItem->addGift(new CartItemGift($gift->getProduct(), $gift->getName()));
			}
		}
	}

	public function cartFillDao(FillDaoEvent $event): void
	{
		if (!\str_starts_with(Module::$currentPresenterName, 'EshopOrders')) {
			return;
		}

		$cart = $event->cart;

		$cartItems = $cart->getCartItemsForGifts();
		if ($cartItems !== []) {
			$itemsPrice = 0;
			foreach ($cartItems as $item) {
				$itemsPrice += $item->getTotalPrice();
			}

			foreach ($this->orderGifts->findGifts($itemsPrice) as $gift) {
				$dao              = new \EshopOrders\FrontModule\Model\Dao\Gift($gift->getId(), $gift->getName());
				$dao->productId   = $gift->getProduct()->getId();
				$dao->product     = $gift->getProduct();
				$dao->code1       = $gift->code1;
				$dao->ean         = $gift->ean;
				$dao->description = $gift->getDescription();

				$cart->addGift($dao);
			}
		}
	}
}
