<?php declare(strict_types = 1);

namespace EshopGifts\Model\Entities;

use DateTime;
use Core\Model\Entities\Site;
use Core\Model\Entities\TId;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\GroupCustomers;

#[ORM\Table('eshop_gifts__order_gift')]
#[ORM\Entity]
class OrderGift
{
	use TId;

	#[ORM\JoinColumn(name: 'gift_id', referencedColumnName: 'id', nullable: false, onDelete: 'CASCADE')]
	#[ORM\ManyToOne(targetEntity: Gift::class)]
	protected Gift $gift;

	/** @var Collection<string, Site> */
	#[ORM\JoinTable(name: 'eshop_gifts__order_gifts')]
	#[ORM\JoinColumn(name: 'gift_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'site_id', referencedColumnName: 'ident', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: Site::class, indexBy: 'ident')]
	protected Collection $sites;

	/** @var Collection<GroupCustomers> */
	#[ORM\JoinTable(name: 'eshop_gifts__order_gift_customer_group')]
	#[ORM\JoinColumn(name: 'og_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'group_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: GroupCustomers::class, indexBy: 'id')]
	public Collection $customerGroups;

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 0, 'unsigned' => true])]
	public int $withoutCustomerGroup = 0;

	/**
	 * @var double|null
	 */
	#[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
	public $priceFrom;

	/**
	 * @var double|null
	 */
	#[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
	public $priceTo;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	public ?DateTime $dateFrom = null;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	public ?DateTime $dateTo = null;

	#[ORM\Column(type: Types::SMALLINT, length: 1, options: ['default' => 1])]
	public int $isActive = 1;

	public function __construct(Gift $gift)
	{
		$this->setGift($gift);
		$this->isActive       = 1;
		$this->sites          = new ArrayCollection;
		$this->customerGroups = new ArrayCollection;
	}

	public function getGift(): Gift { return $this->gift; }

	public function setGift(Gift $gift): self
	{
		$this->gift = $gift;

		return $this;
	}

	public function isActive(): int { return (int) $this->isActive; }

	/**
	 * @param bool|int $active
	 */
	public function setActive($active): self
	{
		$this->isActive = (int) $active;

		return $this;
	}

	/**
	 * @return Collection<string, Site>
	 */
	public function getSites() { return $this->sites; }

	public function setSites(array $sites): self
	{
		foreach (array_diff($this->sites->getKeys(), array_keys($sites)) as $v) {
			$this->sites->remove($v);
		}

		foreach (array_diff(array_keys($sites), $this->sites->getKeys()) as $v) {
			$this->sites->add($sites[$v]);
		}

		return $this;
	}

	/**
	 * @return Collection<int, GroupCustomers>
	 */
	public function getCustomerGroups() { return $this->customerGroups; }

	public function setCustomerGroups(array $groups): self
	{
		foreach (array_diff($this->customerGroups->getKeys(), array_keys($groups)) as $v) {
			$this->customerGroups->remove($v);
		}

		foreach (array_diff(array_keys($groups), $this->customerGroups->getKeys()) as $v) {
			$this->customerGroups->add($groups[$v]);
		}

		return $this;
	}
}
