<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Subscribers;

use Contributte\EventDispatcher\EventSubscriber;
use Core\Model\Mailing\TemplateFactory;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Localization\ITranslator;
use Nette\Mail\IMailer;
use Nette\Mail\Message;

class OrderMailerSubscriber implements EventSubscriber
{
	/** @var  array */
	protected $conf;

	/** @var ITranslator */
	protected $translator;

	/** @var TemplateFactory */
	protected $templateFactory;

	/** @var IMailer */
	protected $mailer;

	/** @var SessionSection */
	protected $orderSuccessMessages;

	public function __construct(array $conf, ITranslator $translator, TemplateFactory $templateFactory, IMailer $mailer, Session $session)
	{
		$this->conf                 = $conf;
		$this->translator           = $translator;
		$this->templateFactory      = $templateFactory;
		$this->mailer               = $mailer;
		$this->orderSuccessMessages = $session->getSection('eshopOrders/orderSuccessMessages');
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderOnSuccess' => ['orderOnSuccess', 100],
		];
	}

	public function orderOnSuccess(OrderEvent $event): void
	{
		if ($this->conf['send'] == false)
			return;

		$order = $event->order;
		$conf  = $this->conf;
		$file  = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/orderSend.latte';
		if (!file_exists($file) || !$conf['fromEmail'] || !$conf['toEmail'])
			return;

		$template = $this->templateFactory->create();

		$subject = $this->translator->translate('eshopOrders.emails.subject', [
			'orderId'  => $order->getId(),
			'siteName' => $template->siteName,
		]);

		$template->setFile($file);
		$template->setParameters([
			'subject' => $subject,
			'order'   => $order,
		]);

		$message = new Message();
		$message->setSubject($subject);
		$message->setFrom($conf['fromEmail'], $conf['fromName'] ?? null);
		$message->addTo($conf['toEmail'], $conf['toName'] ?? null);

		if (isset($conf['bcc']) && $conf['bcc']) {
			foreach (explode(',', $conf['bcc']) as $v) {
				$message->addBcc(trim($v));
			}
		}

		$message->setHtmlBody($template->renderToString());

		$message->addTo($order->getAddressDelivery()->getEmail(), $order->getAddressDelivery()->getFirstName() . ' ' . $order->getAddressDelivery()->getLastName());

		$this->mailer->send($message);
		$this->orderSuccessMessages->success[] = 'eshopOrders.orderPage.sent';
	}
}
