<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Subscribers;

use Contributte\EventDispatcher\EventSubscriber;
use Core\Model\Event\PresenterTemplateEvent;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\Tags;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use EshopOrders\FrontModule\Model\Event\RemovedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use Mpdf\Tag;
use Nette\Localization\ITranslator;

class CartSubscriber implements EventSubscriber
{
	/** @var Carts */
	protected $cartsService;

	/** @var Products */
	protected $productsService;

	/** @var Tags */
	protected $tagsService;

	/** @var ITranslator */
	protected $translator;

	/** @var array */
	protected $cartChanges = [];

	public function __construct(Carts $carts, Products $products, Tags $tags, ITranslator $translator)
	{
		$this->cartsService    = $carts;
		$this->productsService = $products;
		$this->tagsService     = $tags;
		$this->translator      = $translator;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'core.front.beforeRender'      => 'beforeRender',
			'eshopOrders.cartAddItem'      => ['onAddItem', 100],
			'eshopOrders.cartUpdateItem'   => ['onUpdateItem', 100],
			'eshopOrders.cartRemoveItem'   => ['onRemoveItem', 100],
			'eshopOrders.cartFillDaoItems' => ['onFillDaoItems', 100],
		];
	}

	public function beforeRender(PresenterTemplateEvent $event): void
	{
		if (!empty($this->cartChanges)) {
			$message = null;
			if (isset($this->cartChanges['added']))
				$message = 'itemAdded';
			if (isset($this->cartChanges['updated']))
				$message = 'itemUpdated';
			if (isset($this->cartChanges['removed']))
				$message = 'itemRemoved';
			if ($message)
				$event->presenter->payload->notify[] = [
					'type'    => 'success',
					'message' => $this->translator->translate('eshopOrdersFront.cart.' . $message),
				];
		}
	}

	public function onAddItem(AddedCartItemEvent $item): void
	{
		$this->cartChanges['added'][] = $item;
		$this->cartsService->addItem($item);
	}

	public function onUpdateItem(UpdatedCartItemEvent $item): void
	{
		$this->cartChanges['updated'][] = $item;
		$this->cartsService->updateItemQuantity($item->itemId, $item->quantity);
	}

	public function onRemoveItem(RemovedCartItemEvent $item): void
	{
		$this->cartChanges['removed'][] = $item;
		$this->cartsService->removeItem($item->itemId);
	}

	public function onFillDaoItems(FillDaoItemsEvent $event): void
	{
		$cartItems = $event->items;
		if ($this->cartsService->cDaoItems)
			return;

		$items = [];
		foreach ($cartItems as $ci) {
			$product = $this->productsService->get($ci->getProductId(), ['loadPrice' => true]);

			if ($product) {
				$this->productsService->loadVariants($product);
				$this->tagsService->loadTagsToProduct($product);
			}

			$variant = null;
			if ($ci->getVariantId()) {
				$variant = $this->productsService->getVariant((int) $ci->getVariantId()); //varianta
			}

			if ($product) {
				$cartItem = new Dao\CartItem((string) $product->getName(), $product->getId(), $product->getPrice());

				$cartItem
					->setId($ci->getId())
					->setCartId($ci->cart->getId())
					->setProductId($ci->getProductId())
					->setProduct($product)
					->setVariantId($ci->getVariantId())
					->setProductVariant($variant)
					->setQuantity($ci->getQuantity());

				$tagFree = $product->getTag('freeDelivery');
				if ($tagFree && !$tagFree->isAuto)
					$cartItem->freeDelivery = true;

				$items[$ci->getId()] = $cartItem;
			}
		}

		$this->cartsService->cDaoItems = $items;
	}
}
