<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Doctrine\ORM\Mapping as ORM;
use Doctrine\Common\Collections\ArrayCollection;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;

/**
 * @ORM\Table("eshop_orders__order_discount")
 * @ORM\Entity
 */
class OrderDiscount implements IDiscount
{
	const TYPE_PERCENT = 'percent';
	const TYPE_FIX     = 'fix';

	use Identifier;
	use MagicAccessors;

	/**
	 * @var string
	 * @ORM\Column(name="code", type="string")
	 */
	protected $code;

	//protected $discount_id //odkaz do tabulky se seznamem pouzitelnych slevovych kodu - pridat pozdeji

	/**
	 * @var string
	 * @ORM\Column(name="name", type="string")
	 */
	protected $name;

	/** sleva v procentech nebo pevna castka
	 * @var int
	 * @ORM\Column(name="value", type="integer")
	 */
	protected $value;

	/** vysledna sleva vypocitana z procent - zaporna hodnota
	 * @var float
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;

	/** procenta / fixni castka
	 * @var string
	 * @ORM\Column(name="type", type="string")
	 */
	protected $type;

	/**
	 * @var Order
	 * @ORM\ManyToOne(targetEntity="Order", inversedBy="orderDiscounts")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $order;

	public function __construct($code, $order)
	{
		$this->code  = $code;
		$this->order = $order;
	}

	public function calculateDiscount(float $price): float
	{
		if ($this->type === self::TYPE_PERCENT)
			$discount = -($price * ((float) $this->value / 100));
		else
			$discount = -((float) $this->value);

		return ceil($discount);
	}

	/*****
	 * === Code
	 */

	public function getCodeFormatted(): string
	{
		if (in_array((string) $this->code, ['ESHOPSALESAUTO']))
			return '';

		return (string) $this->code;
	}

	public function getCode()
	{
		return $this->code;
	}

	public function setCode($code): OrderDiscount
	{
		$this->code = $code;

		return $this;
	}

	/*****
	 * === Name
	 */

	public function getName()
	{
		return $this->name;
	}

	public function setName($name): OrderDiscount
	{
		$this->name = $name;

		return $this;
	}

	/*****
	 * === Quantity
	 */

	public function setQuantity($quantity)
	{
		$this->quantity = $quantity && $quantity > 0 ? $quantity : null;
	}

	public function getQuantity()
	{
		return $this->quantity;
	}

	/*******
	 * === Value
	 */

	public function getValue(): float
	{
		return (float) $this->value;
	}

	public function setValue(float $value)
	{
		$this->value = $value;

		return $this;
	}

	/*******
	 * === Price
	 */

	public function getPrice(): float
	{
		return (float) $this->price;
	}

	public function setPrice(float $price)
	{
		$this->price = $price;

		return $this;
	}

	/*******
	 * === Type
	 */

	public function getType(): string
	{
		return $this->type;
	}

	public function setType($type): OrderDiscount
	{
		$this->type = $type;

		return $this;
	}

	/*******
	 * === Order
	 */

	public function getOrder(): Order
	{
		return $this->order;
	}

	public function setOrder(Order $order)
	{
		$this->order = $order;

		return $this;
	}
}

