<?php declare(strict_types=1);

namespace EshopOrders\AdminModule\Components\Invoice;

use Core\Model\Pdf\Pdf;
use Core\Model\Templating\Filters\Price;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Invoices;
use Nette\Utils\Html;
use Ublaboo\DataGrid\Exception\DataGridException;

class InvoicesGrid extends BaseControl
{
	/** @var Invoices */
	protected $invoices;

	/** @var Price */
	protected $priceFilter;

	/**
	 * InvoiceConfigForm constructor.
	 * @param Invoices $invoices
	 * @param Price $priceFilter
	 */
	public function __construct(Invoices $invoices, Price $priceFilter)
	{
		parent::__construct();
		$this->invoices = $invoices;
		$this->priceFilter = $priceFilter;
	}

	/**
	 * @return BaseDataGrid
	 * @throws DataGridException
	 */
	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$grid->setDataSource($this->invoices->getQueryBuilder());

		// columns
		$grid->addColumnText('ident', 'eshopOrders.invoicesGrid.ident');
		$grid->addColumnText('subscriber', 'eshopOrders.invoicesGrid.subscriber')->setRenderer(static function (Invoice $invoice) {
			$customer = $invoice->invoiceData->getCustomer();
			return $customer->company ?? sprintf('%s %s', $customer->firstName, $customer->lastName);
		});
		$grid->addColumnDateTime('fromDate', 'eshopOrders.invoicesGrid.fromDate')->setRenderer(static function (Invoice $invoice) {
			return $invoice->createdAt->format('d. m. Y');
		});
		$grid->addColumnDateTime('dueDate', 'eshopOrders.invoicesGrid.dueDate');
		$grid->addColumnText('total', 'eshopOrders.invoicesGrid.totalPrice')->setRenderer(function (Invoice $invoice) {
			return $this->priceFilter->format($invoice->invoiceData->getPrice());
		});

		// actions
		$grid->addAction('pdf', '')->setRenderer(function (Invoice $invoice) {
			return Html::el('a', ['href' => $this->presenter->link('pdf!', $invoice->getId()), 'class' => 'btn btn-danger btn-xs', 'target' => '_blank'])
					   ->addHtml(Html::el('i', ['class' => 'fas fa-file-pdf']))
					   ->addHtml('&nbsp;')
					   ->addText($this->t('eshopOrders.invoicesGrid.pdf'));
		});

		$grid->addAction('regenerate', '', 'regenerate!', ['invoiceId' => 'id'])
			 ->addClass('ajax')
			 ->setIcon('fa fa-sync')
			 ->setTitle('eshopOrders.invoicesGrid.actions.regenerate')
			 ->setConfirm('eshopOrders.invoicesGrid.actions.regenerateConfirm');

		return $grid;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @param mixed $invoiceId
	 */
	public function handleRegenerate($invoiceId): void
	{
		if ($this->invoices->regenerateInvoice((int) $invoiceId)) {
			$this->presenter->flashMessageSuccess('eshopOrders.invoicesGrid.actions.regenerateSuccess');
		} else {
			$this->presenter->flashMessageDanger('eshopOrders.invoicesGrid.actions.regenerateDanger');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

}