<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Application\UI\ITemplateFactory;
use Nette\Utils\ArrayHash;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Statuses;
use Users\Model\Users;

class OrderStatusForm extends BaseControl
{
	/** @var OrderStatus */
	public $orderStatus;

	/** @var Order */
	public $order;

	/** @var Orders */
	protected $ordersService;

	/** @var Users */
	protected $usersService;

	/** @var Statuses */
	protected $statusesService;

	/** @var ITemplateFactory */
	protected $templateFactory;

	public function __construct($data, ITemplateFactory $templateFactory, Orders $orders, Users $users, Statuses $statuses)
	{
		$this->ordersService   = $orders;
		$this->usersService    = $users;
		$this->statusesService = $statuses;

		$this->templateFactory = $templateFactory;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Setters
	 */

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$statuses   = $this->statusesService->getAll();
		$statusList = [];
		foreach ($statuses as $status) {
			$id = $status->getId();
			if ($id != 'created') {
				$statusList[$id] = $status->getName();
			}
		}

		//TODO defaultne nevybrany zadny status, a validace na to ze musi byt vybrany
		$form->addSelect('status', 'eshopOrders.default.status', $statusList);
		$form->addTextArea('message', 'eshopOrders.default.statusMessage');

		$form->addSaveCancelControl();
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$order   = $this->order;
			$status  = $this->statusesService->getReference($values->status);
			$userRef = $this->usersService->getReference($this->getPresenter()->getUser()->id);

			$orderStatus = new OrderStatus($order, $status, $userRef);
			$orderStatus->setMessage($values->message);
			$this->em->persist($orderStatus)->flush();

			$this->getPresenter()->flashMessageSuccess('eshopOrders.statusForm.added');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		try {
			$this->statusesService->sendOrderStatusEmail($order, $orderStatus);
			$this->getPresenter()->flashMessageSuccess('eshopOrders.statusForm.emailSent');
		} catch (\Exception $e) {
			$this->getPresenter()->flashMessageDanger('eshopOrders.statusForm.emailError');
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setOrder($orderId)
	{
		$this->order = $this->ordersService->get($orderId);
	}
}
