<?php declare(strict_types=1);

namespace EshopOrders\AdminModule\Components\Invoice;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\Model\Entities\NumericalSeries;
use EshopOrders\Model\InvoiceConfigRepository;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Utils\Strings;
use Exception;
use Nette\Forms\IControl;
use Nette\Utils\ArrayHash;

class InvoiceConfigForm extends BaseControl
{
	/** @var InvoiceConfigRepository */
	protected $invoiceConfigRepository;

	/** @var Invoices */
	protected $invoices;

	/**
	 * InvoiceConfigForm constructor.
	 * @param InvoiceConfigRepository $invoiceConfigRepository
	 * @param Invoices $invoices
	 */
	public function __construct(InvoiceConfigRepository $invoiceConfigRepository, Invoices $invoices)
	{
		parent::__construct();
		$this->invoiceConfigRepository = $invoiceConfigRepository;
		$this->invoices = $invoices;
	}

	/**
	 * @return BaseForm
	 */
	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addInteger('maturity', 'eshopOrders.invoiceConfigForm.maturity')
			 ->setRequired()
			 ->addRule($form::MIN, null, 1);

		$form->addGroup('eshopOrders.invoiceConfigForm.numericalSeries.default.caption');

		$form->addText('prefix', 'eshopOrders.invoiceConfigForm.numericalSeries.default.prefix')
			 ->setRequired()
			 ->addRule(static function (IControl $control): bool { // If prefix contains month wildcards then it must also contain year wildcards
			 	$value = $control->getValue();
			 	return !(Strings::containsMore($value, NumericalSeries::POSSIBLE_PREFIX_WILDCARDS['month']) &&
					!Strings::containsMore($value, NumericalSeries::POSSIBLE_PREFIX_WILDCARDS['year']));
			 }, 'eshopOrders.invoiceConfigForm.numericalSeries.errors.prefixValue');

		$form->addInteger('digitsCount', 'eshopOrders.invoiceConfigForm.numericalSeries.default.digitsCount')
			 ->setRequired()
			 ->addRule($form::MIN, null, 1);

		$form->addInteger('startNumber', 'eshopOrders.invoiceConfigForm.numericalSeries.default.startNumber')
			 ->setRequired();

		$form->addSubmit('submit', 'default.save');

		$form->onAnchor[] = function (BaseForm $form) {

			// we cannot change numerical series if invoices count is bigger than 0
			if ($this->invoices->getIvoicesCount() > 0) {
				$form['prefix']->setDisabled();
				$form['digitsCount']->setDisabled();
				$form['startNumber']->setDisabled();
			}

			$this->setDefaults();
		};

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	/**
	 * @param BaseForm $form
	 * @param ArrayHash $values
	 * @throws Exception
	 */
	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$charsLength = 10;
		if ((Strings::length($values->prefix) + $values->digitsCount) > $charsLength) { // prefix + digits count char length must not exceed 10, becouse variable symbol is up to 10 chars
			$form->addError($this->t('eshopOrders.invoiceConfigForm.numericalSeries.errors.charsLength', null, ['length' => $charsLength]));
		} else if (Strings::length($values->startNumber) > $values->digitsCount) {
			$form->addError('eshopOrders.invoiceConfigForm.numericalSeries.errors.maxStartNumber');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	/**
	 * @param BaseForm $form
	 * @param ArrayHash $values
	 * @throws Exception
	 */
	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		$defaultConfig = $this->invoiceConfigRepository->getDefaultConfig();
		$defaultConfig->maturity = $values->maturity;
		$defaultConfig->numericalSeries->prefix = $values->prefix ?? $defaultConfig->numericalSeries->prefix;
		$defaultConfig->numericalSeries->startNumber = $values->startNumber ?? $defaultConfig->numericalSeries->startNumber;
		$defaultConfig->numericalSeries->digitsCount = $values->digitsCount ?? $defaultConfig->numericalSeries->digitsCount;

		$this->invoiceConfigRepository->save($defaultConfig);
		$this->presenter->flashMessageSuccess('default.saved');
		$this->presenter->redrawControl('flashes');
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @throws Exception
	 */
	private function setDefaults(): void
	{
		$defaultConfig = $this->invoiceConfigRepository->getDefaultConfig();
		$form = $this['form'];
		$form->setDefaults([
			'maturity' => $defaultConfig->maturity,
			'prefix' => $defaultConfig->numericalSeries->prefix,
			'digitsCount'=>$defaultConfig->numericalSeries->digitsCount,
			'startNumber'=>$defaultConfig->numericalSeries->startNumber
		]);
	}

}