<?php declare(strict_types=1);

namespace EshopOrders\Model\Entities\Invoice;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\Invoice;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;

/**
 * @ORM\Table("eshop_orders__invoice_data")
 * @ORM\Entity
 */
class InvoiceData
{
	use Identifier;
	use MagicAccessors;

	/**
	 * @var ArrayCollection|Product[]
	 *
	 * @ORM\OneToMany(targetEntity="Product", mappedBy="invoiceData", cascade={"persist"})
	 * @ORM\OrderBy({"name" = "ASC"})
	 */
	public $products;

	/**
	 * @var Customer
	 * @ORM\OneToOne(targetEntity="Customer", cascade={"persist"})
	 * @ORM\JoinColumn(name="customer_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $customer;

	/**
	 * @var ArrayCollection|Discount[]
	 *
	 * @ORM\OneToMany(targetEntity="Discount", mappedBy="invoiceData", cascade={"persist"})
	 */
	public $discounts;

	/**
	 * @var string
	 * @ORM\Column(type="string")
	 */
	public $lang;

	/**
	 * @var string
	 * @ORM\Column(type="string")
	 */
	public $currency;

	/**
	 * @var Supplier
	 * @ORM\OneToOne(targetEntity="Supplier", cascade={"persist"})
	 * @ORM\JoinColumn(name="supplier_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $supplier;

	/**
	 * @var Spedition
	 * @ORM\OneToOne(targetEntity="Spedition", cascade={"persist"})
	 * @ORM\JoinColumn(name="spedition_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $spedition;

	/**
	 * @var Payment
	 * @ORM\OneToOne(targetEntity="Payment", cascade={"persist"})
	 * @ORM\JoinColumn(name="payment_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $payment;

	/**
	 * @var Invoice
	 * @ORM\OneToOne(targetEntity="EshopOrders\Model\Entities\Invoice", cascade={"persist"})
	 * @ORM\JoinColumn(name="invoice_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $invoice;

	/**
	 * InvoiceData constructor.
	 * @param Customer $customer
	 * @param Supplier $supplier
	 * @param Spedition $spedition
	 * @param Payment $payment
	 * @param Invoice $invoice
	 */
	public function __construct(Customer $customer, Supplier $supplier, Spedition $spedition, Payment $payment, Invoice $invoice)
	{
		$this->customer = $customer;
		$this->supplier = $supplier;
		$this->spedition = $spedition;
		$this->payment = $payment;
		$this->invoice = $invoice;
	}

	/**
	 * @return Customer
	 */
	public function getCustomer(): Customer
	{
		return $this->customer;
	}

	/**
	 * @return Supplier
	 */
	public function getSupplier(): Supplier
	{
		return $this->supplier;
	}

	/**
	 * @return Spedition
	 */
	public function getSpedition(): Spedition
	{
		return $this->spedition;
	}

	/**
	 * @return Payment
	 */
	public function getPayment(): Payment
	{
		return $this->payment;
	}

	public function getPrice()
	{
		$priceTotal = $this->getPriceItems();
		$priceTotal += $this->spedition->price;
		$priceTotal += $this->payment->price;
		if ($this->discounts) {
			foreach ($this->discounts as $discount) {
				$priceTotal += $discount->calculateDiscount($priceTotal);
			}
		}

		return $priceTotal;
	}

	public function getPriceItems()
	{
		$priceTotal = 0;
		foreach ($this->products as $item) {
			$priceTotal += $item->getPrice() * $item->getQuantity();
		}

		return $priceTotal;
	}

	public function getPriceItemsDiscount()
	{
		$priceTotal = $this->getPriceItems();
		if ($this->discounts) {
			foreach ($this->discounts as $discount) {
				$priceTotal += $discount->getPrice();
			}
		}

		return $priceTotal;
	}

	public function getPriceWithoutVat()
	{
		$priceTotal = $this->calculateDiscounts($this->getItemsPriceWithoutVat());

		if ($this->spedition)
			$priceTotal += $this->spedition->price;
		if ($this->payment)
			$priceTotal += $this->payment->price;

		return $priceTotal;
	}

	public function getItemsPriceWithoutVat(): float
	{
		$priceTotal = 0;
		foreach ($this->products as $item) {
			$priceTotal += $item->getTotalPriceWithoutVat();
		}

		return $priceTotal;
	}

	public function getPaySpedPrice()
	{
		$priceTotal = 0;
		$priceTotal += $this->spedition->price;
		$priceTotal += $this->payment->price;

		return $priceTotal;
	}

	public function calculateDiscounts(float $price): float
	{
		foreach ($this->discounts as $discount) {
			$price += $discount->calculateDiscount($price);
		}

		return $price;
	}


}