<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Application\UI\Form;
use Nette\Application\UI\ITemplateFactory;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Nette\Mail\IMailer;
use Nette\Mail\Message;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderItemTexts;
use EshopOrders\Model\Orders;
use EshopOrders\Model\OrderItems;
use EshopCatalog\AdminModule\Model\Products;

class OrderItemAddForm extends BaseControl
{
	
	/** @var Order */
	public $order;
	
	/** @var Orders */
	protected $ordersService;
	
	/** @var Items */
	protected $orderItemsService;
	
	/** @var Products */
	protected $productsService;
	
	public function __construct(Orders $orders, OrderItems $items, Products $products)
	{
		$this->ordersService     = $orders;
		$this->orderItemsService = $items;
		$this->productsService   = $products;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}
	
	public function attached($presenter)
	{
		parent::attached($presenter);
	}
	
	/*******************************************************************************************************************
	 * ======================== Handler
	 */
	
	/**
	 * @param int $itemId
	 */
	public function handleProductAutocomplete()
	{
		$term = $this->getParameter('term');
		
		$products = $this->productsService->getByTerm($term);
		$output = [];
		foreach ($products as $product) {
			$output[$product->getId()] = array(
				'id' => $product->getId(),
				'name' => $product->getProductText($this->translator->getLocale())->getName() ." - Kód 1: ". $product->code1,
				'code1' => $product->code1,
				'price' => $product->price,
				'vatRate' => $product->getVateRate()->rate,
			);
		}
		
		$this->getPresenter()->sendJson($output);
	}
	
	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		
		$form->addHidden('productId');
		//$form->addHidden('variant_id');
		$form->addText('product', 'eshopOrders.default.productId')->setHtmlAttribute('class', ['productAutocomplete', 'typeahead'])->setRequired();
		//$form->addInteger('variant', 'eshopOrders.default.variantId')->setRequired();
		$form->addInteger('quantity', 'eshopOrders.default.quantity')->setRequired()->addRule(Form::MIN, 'Počet musí být alespoň %d', 1);
		$form->addText('code1', 'eshopOrders.itemsGrid.code1')->setRequired();
		$form->addText('price', 'eshopOrders.default.pricePiece')->setRequired()->addRule(Form::FLOAT, 'Cena musí být číslo');
		$form->addText('vatRate', 'eshopOrders.itemsGrid.vatRate')->setRequired()->addRule(Form::INTEGER, 'DPH musí být celé číslo');

		$form->addSaveCancelControl();
		$form->onSuccess[] = [$this, 'formSuccess'];
		
		$form->setDefaults([
			'quantity' => 1
		]);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$orderItem = new OrderItem($values->productId, $this->order);
			$orderItem->setCode1($values->code1);
			$orderItem->addOrderItemText('cs');
			$orderItem->getOrderItemText('cs')->setName($values->product);
			$orderItem->getOrderItemText('cs')->setVariantName('');
			
			$flashMessage = 'eshopOrders.itemForm.added';
			
			$orderItem->setQuantity($values->quantity);
			$orderItem->setPrice($values->price);
			$orderItem->setVatRate($values->vatRate);
			
			$this->em->persist($orderItem)->flush();
			
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}
	}
	
	public function setOrder($orderId)
	{
		$this->order   = $this->ordersService->get($orderId);
	}
}
