<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Helpers\BaseService;
use EshopOrders\Model\Entities\OrderDiscount;
use Nette\SmartObject;
use Nette\Http\Session;
use Nette\Http\SessionSection;

/**
 *
 * class CartHelper
 * @package EshopOrders\Model
 *
 */
class CartHelper extends BaseService
{
	use SmartObject;
	/**
	 * @var Carts
	 */
	protected $cartsService;
	/**
	 * @var Speditions
	 */
	protected $speditionsService;

	/** @var string */
	public $discountCode;

	/** @var float */
	public $discountValue;

	/** @var float */
	public $discountMinOrder;

	/** @var SessionSection */
	protected $sessionSection;

	public function __construct($discountData, Carts $carts, Speditions $speditions, Session $session)
	{
		$this->cartsService      = $carts;
		$this->speditionsService = $speditions;

		$this->discountCode     = $discountData['discountCode'];
		$this->discountValue    = $discountData['discountValue'];
		$this->discountMinOrder = $discountData['discountMinOrder'];

		$this->sessionSection = $session->getSection('eshopOrdersOrderForm');
	}

	// --------------- Doprava zdarma ------------

	public function countFreeSpedition()
	{
		$freeSpeditionPrice = $remainingPrice = null;

		$firstFreeSpedition = $this->speditionsService->getFirstFreeSpedition();
		if ($firstFreeSpedition) {
			$freeSpeditionPrice = $firstFreeSpedition->getFreeFrom();

			$cart      = $this->cartsService->getCurrentCart();
			$cartPrice = $cart->getCartItemsPrice();

			if ($cartPrice > $freeSpeditionPrice) {
				$remainingPrice = 0;
			} else {
				$remainingPrice = $freeSpeditionPrice - $cartPrice;
			}
		}

		$cart = $this->cartsService->getCurrentCart();
		foreach ($cart->getCartItems() as $item) {
			if ($item->freeDelivery === true) {
				$freeSpeditionPrice = $item->price; //ne 0 -> to by znamenalo ze neni zdarma nikdy
				$remainingPrice     = 0;
				break;
			}
		}

		return ['minPrice' => $freeSpeditionPrice, 'remaining' => $remainingPrice];
	}

	// --------------- Slevove kody ------------

	public function checkValidDiscountCode($code)
	{
		return ($code == $this->discountCode);
	}

	public function getDiscountTypeText($type)
	{
		return (($type == OrderDiscount::TYPE_PERCENT) ? ' %' : ' Kč');
	}

	public function countDiscountPrice($cartValue, $type = OrderDiscount::TYPE_PERCENT)
	{
		if ($type == OrderDiscount::TYPE_FIX) {
			$price = -$this->discountValue;
		} else {
			$discount = round($cartValue * $this->discountValue / 100);
			$price    = -$discount;
		}

		return $price;
	}

	public function getDiscountCoupon($cartValue)
	{
		//kdyz je moc mala cena objednavky, odebereme kod z kosiku
		if ($cartValue <= $this->discountMinOrder && $this->sessionSection->discountCode) {
			unset($this->sessionSection->discountCode);
		}

		if ($this->sessionSection->discountCode) {
			$type           = OrderDiscount::TYPE_PERCENT;
			$discountCoupon = [
				'discountCode'       => $this->sessionSection->discountCode,
				'discountValue'      => $this->discountValue,
				'discountTypeString' => $this->getDiscountTypeText($type),
				'discountPrice'      => $this->countDiscountPrice($cartValue, $type),
			];

			return $discountCoupon;
		}
	}

}

