<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Contributte\EventDispatcher\EventDispatcher;
use Core\Model\Helpers\BaseFrontEntityService;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\PersistentCollection;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\FillDaoEvent;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use EshopOrders\Model\Entities\Cart;
use EshopOrders\Model\Entities\CartItem;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\FrontModule\Model\Dao\Product;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\SmartObject;
use EshopCatalog\Model\Config as EshopCatalogConfig;

/**
 * class Carts
 * @package EshopOrders\Model
 *
 */
class Carts extends BaseFrontEntityService
{
	use SmartObject;

	/** @var SessionSection */
	private $sessionSection;

	protected $entityClass = Cart::class;

	/** @var Dao\Cart[]|null */
	protected $cartsCached;

	/** @var Products */
	private $productsService;

	/** @var Tags */
	protected $tagsService;

	/** @var EventDispatcher */
	protected $eventDispatcher;

	/** @var array */
	public $cDaoItems;

	public function __construct(Session $session, Products $products, Tags $tags, EventDispatcher $eventDispatcher)
	{
		$this->sessionSection  = $session->getSection('eshopOrdersCart');
		$this->productsService = $products;
		$this->tagsService     = $tags;
		$this->eventDispatcher = $eventDispatcher;
		$this->sessionSection->setExpiration('1 month');
	}

	/**
	 *
	 * @param int $itemId
	 *
	 * @return Dao\CartItem|null
	 */
	public function getCartItem($itemId): ?Dao\CartItem
	{
		return $this->getCurrentCart()->getCartItems()[$itemId] ?? null;
	}

	public function getCurrentCartId(): int
	{
		$cartId = $this->sessionSection->cartId;

		if (is_null($cartId)) {
			$cartRaw = $this->createRawCart();
			$cartId  = $cartRaw->getId();

			$this->sessionSection->cartId = $cartId;
		}

		return $cartId;
	}

	/** Vrati soucasny kosik podle ID ze session. Kdyz kosik neexistuje, vytvori novy
	 * @return Dao\Cart
	 */
	public function getCurrentCart()
	{
		$cartId = $this->getCurrentCartId();

		if (!is_null($cartId)) {
			if (isset($this->cartsCached[$cartId]))
				return $this->cartsCached[$cartId];

			$cartRaw = $this->loadRawCart($cartId);
		}

		$cartId                     = $cartRaw->getId();
		$this->cartsCached[$cartId] = $this->fillDao($cartRaw);

		return isset($this->cartsCached[$cartId]) ? ($this->cartsCached[$cartId] ?: null) : null;
	}

	/** smaze kosik pro daneho uzivatele - po dokonceni objednavky
	 * pak, pri volani getCurrentCart se vygeneruje novy
	 */
	public function deleteCurrentCart()
	{
		$oldCartId = $this->sessionSection->cartId;
		$this->removeRawCart($oldCartId);
		$this->sessionSection->cartId = null;
		unset($this->cartsCached[$oldCartId]);
	}

	/** nacte kosik z databaze do vlastni property
	 *
	 * @param int $cartId
	 *
	 * @return Cart
	 */
	private function loadRawCart($cartId): Cart
	{
		$cartRawQuery = $this->getEr()->createQueryBuilder('c', 'c.id');
		$cartRawQuery->andWhere('c.id = :id')->setParameter('id', $cartId);

		$cartRaw = $cartRawQuery->getQuery()->getOneOrNullResult();

		if (is_null($cartRaw)) {
			$cartRaw = $this->createRawCart();

			$this->sessionSection->cartId = $cartRaw->getId();
		}

		return $cartRaw;
	}

	/**
	 * @return Cart
	 */
	private function createRawCart()
	{
		$ident = substr(md5((string) rand()), 0, 7);
		$cart  = new Cart($ident);
		$this->em->persist($cart)->flush($cart);

		return $cart;
	}

	/**
	 * @param $cartId id kosiku ke smazani
	 */
	private function removeRawCart($cartId)
	{
		$cartRaw = $this->getEr()->getReference($cartId);
		$this->em->remove($cartRaw)->flush();
	}

	/** nacte jednu polozku kosiku z databaze
	 *
	 * @param int $itemId
	 *
	 * @return CartItem
	 */
	private function loadRawCartItem($itemId)
	{
		$cartRawQuery = $this->em->getRepository(CartItem::class)->createQueryBuilder('ci', 'ci.id');
		$cartRawQuery->andWhere('ci.id = :item_id')->setParameter('item_id', $itemId);

		$cartItemRaw = $cartRawQuery->getQuery()->getOneOrNullResult();

		return $cartItemRaw;
	}

	public function addDiscount()
	{
		$cartId  = $this->getCurrentCartId();
		$cartRaw = $this->loadRawCart($cartId);
	}

	public function addItem(AddedCartItemEvent $event)
	{
		$productId = (int) $event->item->productId;
		$quantity  = $this->checkProductQuantity($productId, (int) $event->item->quantity);
		$optionId  = $event->item->optionId;
		$itemIdent = $productId . '|' . $optionId; //pripadne dalsi vlastnosti s vlivem na cenu

		$cartId = $this->getCurrentCartId();

		$cartRaw = $this->loadRawCart($cartId);

		$itemRaw = $cartRaw->getCartItems()->get($itemIdent);

		if ($itemRaw) {
			//kdyz polozka uz je v kosiku, jen zvysime pocet
			$itemRaw->quantity = $itemRaw->quantity + $quantity;
		} else {
			//kdyz zatim neexistuje, pridame ji do kosiku
			$itemRaw = new CartItem($itemIdent, $productId, $cartRaw);
			$itemRaw->setVariantId($optionId);
			$itemRaw->setQuantity($quantity);
		}

		$this->em->persist($itemRaw)->flush();

		//vymazeme kosik z cache, aby se priste obnovil z DB.
		//Alternativni moznost - vzit soucasne DAO a pridat mu polozku
		$this->cDaoItems = null;
		unset($this->cartsCached[$cartId]);
		//TODO vratit nejaky priznak, ze to probehlo v poradku?
	}

	public function updateItemQuantity($itemId, $quantity)
	{
		if ($quantity <= 0) {
			return $this->removeItem($itemId);
		}
		$cartId = $this->getCurrentCartId();

		$itemRaw = $this->loadRawCartItem($itemId);
		$itemRaw->setQuantity($this->checkProductQuantity((int) $itemRaw->getProductId(), (int) $quantity));
		$this->em->persist($itemRaw)->flush();

		$this->cDaoItems = null;
		unset($this->cartsCached[$cartId]); //obnoveni cache
	}

	public function removeItem($itemId)
	{
		$cartId = $this->getCurrentCartId();

		$itemRaw = $this->loadRawCartItem($itemId);
		if ($itemRaw) {
			//			unset($this->cDaoItems[$itemId]);
			$this->em->remove($itemRaw)->flush();
		}

		$this->cDaoItems = null;
		unset($this->cartsCached[$cartId]); //obnoveni cache
	}

	public function clearCartItems()
	{
		$cartId = $this->getCurrentCartId();
		$cart   = $this->loadRawCart($cartId);

		foreach ($cart->cartItems->toArray() as $item) {
			$this->em->remove($item);
		}

		$this->em->flush();
		$this->cDaoItems = null;
		unset($this->cartsCached[$cartId]);
	}

	protected function checkProductQuantity(int $productId, int $quantity): int
	{
		if (EshopCatalogConfig::load('pseudoWarehouse')) {
			$product = $this->productsService->get($productId);

			if ((int) $product->getExtraField('unlimitedQuantity', '0') === 0 && $quantity > $product->getQuantity()) {
				$quantity = $product->getQuantity();
			}
		}

		return $quantity;
	}

	/**
	 * @param Cart $cartRaw
	 *
	 * @return Dao\Cart
	 */
	protected function fillDao($cartRaw)
	{
		$cart = new Dao\Cart();
		$cart
			->setId($cartRaw->getId())
			->setIdent($cartRaw->ident)
			->setCartItems($this->fillDaoItems($cartRaw->getCartItems()));

		$this->eventDispatcher->dispatch('eshopOrders.cartFillDao', new FillDaoEvent($cart, $cartRaw));

		return $cart;
	}

	/**
	 * @param Entity\CartItem[] $cartItemsRaw
	 *
	 * @return ArrayCollection(Dao\CartItem) polozky
	 */
	protected function fillDaoItems($cartItemsRaw)
	{
		if ($cartItemsRaw instanceof ArrayCollection || $cartItemsRaw instanceof PersistentCollection)
			$cartItemsRaw = $cartItemsRaw->toArray();

		$this->eventDispatcher->dispatch('eshopOrders.cartFillDaoItems', new FillDaoItemsEvent($cartItemsRaw));

		return $this->cDaoItems;
	}

}

