<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\Common\Collections\ArrayCollection;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderItemTexts;
use EshopOrders\Model\Entities\Status;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use Nette\Localization\ITranslator;

/**
 * class Orders
 * @package EshopOrders\Model
 * 
 * @method Order|object|null getReference($id)
 * @method Order[]|null getAll()
 * @method Order|null get($id)
 */
class Orders extends BaseEntityService
{
	const ADDRESS_DELIVERY = 'delivery';
	const ADDRESS_INVOICE = 'invoice';
	
	protected $entityClass = Order::class;

	/** @var ITranslator @inject */
	public $translator;
	
	public function __construct()
	{
	}
	
	/** Vrati seznam objednavek podle zakaznika
	 * @return ArrayCollection|Order[]
	 */
	public function getByCustomer($customer = null)
	{
		if(is_null($customer)) {
			return new ArrayCollection();
		}
		$orderRawQuery = $this->getEr()->createQueryBuilder('o', 'o.id');
		$orderRawQuery->andWhere('o.customer = :customer')->setParameter('customer', $customer);
		$orderRawQuery->orderBy('o.id', 'DESC');
		
		$orders = $orderRawQuery->getQuery()->getResult();
		
		return $orders;
	}
	
	public function getByIdent($ident)
	{
		if(is_null($ident)) {
			return null;
		}
		$orderQuery = $this->getEr()->createQueryBuilder('o', 'o.id');
		$orderQuery->andWhere('o.ident = :ident')->setParameter('ident', $ident);
		
		$order = $orderQuery->getQuery()->getOneOrNullResult();
		
		return $order;
	}
	
	/**
	 * 
	 * @param CartItem[] $itemsCart
	 * @param Order $order
	 * @return OrderItem[]
	 */
	public function fillOrderItems($itemsCart, $order)
	{
		$items = [];
		foreach($itemsCart as $ci) {
			$product = $ci->getProduct();
			$item = new OrderItem($ci->productId, $order);
			
			$item->setVariantId($ci->getVariantId());
			$item->setQuantity($ci->getQuantity());
			$item->setPrice($ci->price);

			if ($product) {
				$item->setVatRate($product->getVatRate());
				$item->setCode1($product->code1);
			}
			
			$item->addOrderItemText($this->translator->getLocale());
			$item->getOrderItemText($this->translator->getLocale())->setName($ci->title);
			
			$variantName = '';
			if($ci->getProductVariant()) {
				$variants = $ci->getProductVariant()->variants;
				$variantName = '';
				
				end($variants);
				$last_key = key($variants);
				reset($variants);
				
				foreach($variants as $key => $variant) {
					$variantName .= $variant['variantName'] . ': ' . $variant['variantValueName'];
					if($key != $last_key) {
						$variantName .= ', ';
					}
				}
			}
			$item->getOrderItemText($this->translator->getLocale())->setVariantName($variantName);
			
			$items[] = $item;
		}
		
		return $items;
	}
}
