<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Model;

use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Orders;
use Kdyby\Doctrine\EntityManager;

class DataConverter
{
	/** @var Orders */
	protected Orders $orders;

	/** @var EntityManager */
	protected EntityManager $em;

	public function __construct(Orders $orders, EntityManager $em)
	{
		$this->orders = $orders;
		$this->em     = $em;
	}

	public function forPrint(array $ids): array
	{
		$statuses = [];
		foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
			         ->select('IDENTITY(os.order) as order, IDENTITY(os.status) as status')
			         ->andWhere('os.order IN (:ids)')
			         ->andWhere('os.deleted = 0')
			         ->setParameter('ids', $ids)
			         ->orderBy('os.created', 'desc')
			         ->getQuery()->getArrayResult() as $row) {
			$statuses[$row['order']][] = [
				'id' => $row['status'],
			];
		}

		$orders = [];
		foreach ($this->orders->getEr()->createQueryBuilder('o')
			         ->addSelect('oItems, oItemTexts, iItemsProd, oDel, oInv, os, op, oDisc, oStat')
			         ->innerJoin('o.orderItems', 'oItems')
			         ->leftJoin('oItems.product', 'iItemsProd')
			         ->leftJoin('oItems.orderItemTexts', 'oItemTexts')
			         ->leftJoin('o.addressDelivery', 'oDel')
			         ->leftJoin('o.addressInvoice', 'oInv')
			         ->innerJoin('o.spedition', 'os')
			         ->innerJoin('o.payment', 'op')
			         ->leftJoin('o.orderDiscounts', 'oDisc')
			         ->innerJoin('o.orderStatuses', 'oStat')
			         ->andWhere('o.id IN (:ids)')
			         ->setParameter('ids', $ids)
			         ->getQuery()->getResult() as $order) {
			/** @var Order $order */
			$row = [
				'id'         => $order->getId(),
				'priceTotal' => $order->getPrice(),
				'name'       => $order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName(),
				'email'      => $order->getAddressInvoice()->getEmail(),
				'phone'      => $order->getAddressInvoice()->getPhone(),
				'created'    => $order->getCreatedTime(),
			];

			$vats  = [
				21 => [
					'base'  => 0,
					'dph'   => 0,
					'total' => 0,
				],
			];
			$items = [];

			// Produkty
			foreach ($order->getOrderItems() as $item) {
				$product = $item->getProduct();
				$items[] = [
					'quantity' => $item->getQuantity(),
					'code1'    => $item->getCode1(),
					'code2'    => $product ? $product->code2 : null,
					'name'     => $item->getOrderItemText()->getName(),
				];

				if (!isset($vats[$item->vatRate])) {
					$vats[$item->vatRate] = [
						'base'  => 0,
						'dph'   => 0,
						'total' => 0,
					];
				}

				$vats[$item->vatRate]['base']  += $item->getTotalPriceWithoutVat();
				$vats[$item->vatRate]['dph']   += round($item->getPriceTotal() - $item->getTotalPriceWithoutVat(), 2);
				$vats[$item->vatRate]['total'] += $item->getPriceTotal();
			}

			// Doprava a platba
			$orderSpedition    = $order->getSpedition();
			$cmsSpedition      = $orderSpedition->getSpedition();
			$orderPayment      = $order->getPayment();
			$items[]           = [
				'quantity' => 1,
				'code1'    => $cmsSpedition ? $cmsSpedition->getIdent() : null,
				'code2'    => null,
				'name'     => trim($orderSpedition->getName() . ' ' . $orderPayment->getName()),
			];
			$vats[21]['base']  += $orderSpedition->getPriceWithoutVat() + $orderPayment->getPriceWithoutVat();
			$vats[21]['dph']   += ($orderSpedition->getPrice() + $orderPayment->getPrice()) - ($orderSpedition->getPriceWithoutVat() + $orderPayment->getPriceWithoutVat());
			$vats[21]['total'] += $orderSpedition->getPrice() + $orderPayment->getPrice();

			$row['items'] = $items;
			$row['vats']  = $vats;
			$orders[]     = $row;
		}

		$vatTotal = [
			'total' => 0,
			'vats'  => [],
		];
		foreach ($orders as $order) {
			foreach ($order['vats'] as $vatRate => $vat) {
				if (!isset($vatTotal['vats'][$vatRate])) {
					$vatTotal['vats'][$vatRate] = [
						'base'  => 0,
						'dph'   => 0,
						'total' => 0,
					];
				}

				$vatTotal['vats'][$vatRate]['base']  += $vat['base'];
				$vatTotal['vats'][$vatRate]['dph']   += $vat['dph'];
				$vatTotal['vats'][$vatRate]['total'] += $vat['total'];

				$vatTotal['total'] += $vat['total'];
			}
		}

		return [
			'statuses' => $statuses,
			'orders'   => $orders,
			'vatTotal' => $vatTotal,
		];
	}
}
