<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Subscribers;

use Contributte\Application\LinkGenerator;
use Contributte\EventDispatcher\EventSubscriber;
use Core\Model\Event\ControlEvent;
use Core\Model\UI\FrontPresenter;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Tags;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use EshopOrders\FrontModule\Model\Event\RemovedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use Nette\Localization\ITranslator;

class CartSubscriber implements EventSubscriber
{
	/** @var Carts */
	protected $cartsService;

	/** @var ProductsFacade */
	protected $productsFacadeService;

	/** @var Tags */
	protected $tagsService;

	/** @var ITranslator */
	protected $translator;

	/** @var LinkGenerator */
	protected $linkGenerator;

	/** @var array */
	protected $cartChanges = [];

	public function __construct(Carts $carts, ProductsFacade $productsFacade, Tags $tags, ITranslator $translator, LinkGenerator $linkGenerator)
	{
		$this->cartsService          = $carts;
		$this->productsFacadeService = $productsFacade;
		$this->tagsService           = $tags;
		$this->translator            = $translator;
		$this->linkGenerator         = $linkGenerator;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FrontPresenter::class . '::beforeRender' => 'beforeRender',
			'eshopOrders.cartAddItem'                => ['onAddItem', 100],
			'eshopOrders.cartUpdateItem'             => ['onUpdateItem', 100],
			'eshopOrders.cartRemoveItem'             => ['onRemoveItem', 100],
			'eshopOrders.cartFillDaoItems'           => ['onFillDaoItems', 100],
		];
	}

	public function beforeRender(ControlEvent $event): void
	{
		/** @var FrontPresenter $presenter */
		$presenter = $event->control;
		if (!empty($this->cartChanges)) {
			$message = null;
			$presenter->redrawControl('orderCartDetail');
			$presenter->redrawControl('cartPreview');
			$presenter->redrawControl('cartPreviewMobile');

			if (isset($this->cartChanges['added']))
				$message = 'itemAdded';
			if (isset($this->cartChanges['updated']))
				$message = 'itemUpdated';
			if (isset($this->cartChanges['removed']))
				$message = 'itemRemoved';
			if ($message)
				$presenter->flashMessageSuccess('eshopOrdersFront.cart.' . $message);

			$presenter->redrawControl('flashes');
		}
	}

	public function onAddItem(AddedCartItemEvent $item): void
	{
		$product = $this->productsFacadeService->getProduct($item->item->productId);
		if (!$product || !$product->canAddToCart)
			return;

		$this->cartChanges['added'][] = $item;
		$this->cartsService->addItem($item);
	}

	public function onUpdateItem(UpdatedCartItemEvent $item): void
	{
		$this->cartChanges['updated'][] = $item;
		$this->cartsService->updateItemQuantity($item->itemId, $item->quantity);
	}

	public function onRemoveItem(RemovedCartItemEvent $item): void
	{
		$this->cartChanges['removed'][] = $item;
		$this->cartsService->removeItem($item->itemId);
	}

	public function onFillDaoItems(FillDaoItemsEvent $event): void
	{
		$cartItems = $event->items;
		if ($this->cartsService->cDaoItems)
			return;

		$items = [];
		foreach ($cartItems as $ci) {
			$product = $this->productsFacadeService->getProduct($ci->getProductId());

			if (!$product || !$product->canAddToCart)
				continue;

			$variant = null;

			if ($product) {
				$cartItem = new Dao\CartItem((string) $product->getName(), $product->getId(), $product->getPrice());

				$cartItem
					->setId($ci->getId())
					->setProductId($ci->getProductId())
					->setProduct($product)
					->setVariantId($ci->getVariantId())
					->setProductVariant($variant)
					->setQuantity($ci->getQuantity())
					->setLink($this->linkGenerator->link('EshopCatalog:Front:Default:product', ['id' => $product->getId()]))
					->setVatRate((int) $product->getVatRate());

				$cartItem->discountDisabled = $product->discountDisabled;

				if ($product->getGallery() && $product->getGallery()->getCover())
					$cartItem->setImage($product->getGallery()->getCover());

				$tagFree = $product->getTag('freeDelivery');
				if ($tagFree && !$tagFree->isAuto)
					$cartItem->freeDelivery = true;

				$items[$ci->getId()] = $cartItem;
			}
		}

		$this->cartsService->cDaoItems = $items;
	}
}
