<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use EshopOrders\Model\Entities\OrderAddress;

/**
 * @ORM\Table("eshop_orders__order")
 * @ORM\Entity
 */
class Order
{
	use Identifier;
	use MagicAccessors;

	/**
	 * @var string
	 * @ORM\Column(name="ident", type="string", length=255, nullable = false)
	 */
	protected $ident;

	/**
	 * @var Customer
	 * @ORM\ManyToOne(targetEntity="Customer")
	 * @ORM\JoinColumn(name="customer_id", referencedColumnName="id", onDelete="SET NULL", nullable = true)
	 */
	protected $customer;

	/**
	 * @var string
	 * @ORM\Column(name="message", type="text", nullable = false)
	 */
	protected $message;

	/**
	 * @var OrderPayment
	 * @ORM\OneToOne(targetEntity="OrderPayment")
	 * @ORM\JoinColumn(name="payment", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $payment;

	/**
	 * @var OrderSpedition
	 * @ORM\OneToOne(targetEntity="OrderSpedition")
	 * @ORM\JoinColumn(name="spedition", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $spedition;

	/**
	 * @var OrderItem[]
	 * @ORM\OneToMany(targetEntity="OrderItem", mappedBy="order", indexBy="id")
	 */
	protected $orderItems;

	/**
	 * @var OrderDiscount[]
	 * @ORM\OneToMany(targetEntity="OrderDiscount", mappedBy="order", indexBy="id")
	 */
	protected $orderDiscounts;

	/**
	 * @var OrderAddress
	 * @ORM\OneToOne(targetEntity="OrderAddress")
	 * @ORM\JoinColumn(name="address_delivery_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected $addressDelivery;

	/**
	 * @var OrderAddress
	 * @ORM\OneToOne(targetEntity="OrderAddress")
	 * @ORM\JoinColumn(name="address_invoice_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected $addressInvoice;

	/**
	 * @var OrderStatus[]
	 * @ORM\OneToMany(targetEntity="OrderStatus", mappedBy="order")
	 */
	protected $orderStatuses;

	/**
	 * @var boolean
	 * @ORM\Column(name="agreed_terms", type="boolean", nullable=true)
	 */
	protected $agreedTerms;

	/**
	 * @var OrderFlag[]
	 * @ORM\OneToMany(targetEntity="OrderFlag", mappedBy="order", indexBy="type")
	 */
	protected $orderFlags;

	/**
	 * @var Invoice|null
	 * @ORM\OneToOne(targetEntity="Invoice", inversedBy="order")
	 * @ORM\JoinColumn(name="invoice_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected $invoice;

	/**
	 * @var string
	 */
	public $currency = 'CZK';

	public function __construct($ident)
	{
		$this->ident           = $ident;
		$this->orderItems      = new ArrayCollection();
		$this->orderDiscounts  = new ArrayCollection();
		$this->orderFlags      = new ArrayCollection();
		$this->addressDelivery = null;
		$this->addressInvoice  = null;
	}

	public function getIdent()
	{
		return $this->ident;
	}

	/**
	 * @return Customer
	 */
	public function getCustomer()
	{
		return $this->customer;
	}

	public function getMessage()
	{
		return $this->message;
	}

	public function getPrice()
	{
		$priceTotal = $this->getPriceItems();
		$priceTotal += $this->spedition->getPrice();
		$priceTotal += $this->payment->getPrice();
		if ($this->orderDiscounts) {
			foreach ($this->orderDiscounts as $discount) {
				$priceTotal += $discount->calculateDiscount($priceTotal);
			}
		}

		return $priceTotal;
	}

	public function getPriceItems()
	{
		$priceTotal = 0;
		foreach ($this->orderItems as $item) {
			$priceTotal += $item->getPrice() * $item->getQuantity();
		}

		return $priceTotal;
	}

	public function getPriceItemsDiscount()
	{
		$priceTotal = $this->getPriceItems();
		if ($this->orderDiscounts) {
			foreach ($this->orderDiscounts as $discount) {
				$priceTotal += $discount->getPrice();
			}
		}

		return $priceTotal;
	}

	public function getPriceWithoutVat()
	{
		$priceTotal = $this->calculateDiscounts($this->getItemsPriceWithoutVat());

		if ($this->spedition)
			$priceTotal += $this->spedition->getPrice();
		if ($this->payment)
			$priceTotal += $this->payment->getPrice();

		return $priceTotal;
	}

	public function getItemsPriceWithoutVat(): float
	{
		$priceTotal = 0;
		foreach ($this->getOrderItems() as $item) {
			$priceTotal += $item->getTotalPriceWithoutVat();
		}

		return $priceTotal;
	}

	public function getItemsVatPrice(): float
	{
		$priceTotal = 0;
		foreach ($this->getOrderItems() as $item) {
			$priceTotal += $item->getPriceTotal() - $item->getTotalPriceWithoutVat();
		}

		return $priceTotal;
	}

	public function getPaySpedPrice()
	{
		$priceTotal = 0;
		$priceTotal += $this->spedition->getPrice();
		$priceTotal += $this->payment->getPrice();

		return $priceTotal;
	}

	/**
	 * @return OrderPayment
	 */
	public function getPayment()
	{
		return $this->payment;
	}

	/**
	 * @return OrderSpedition
	 */
	public function getSpedition()
	{
		return $this->spedition;
	}

	/**
	 * @return OrderItem[]
	 */
	public function getOrderItems()
	{
		return $this->orderItems;
	}

	/**
	 * @return OrderAddress|null
	 */
	public function getAddressDelivery()
	{
		return $this->addressDelivery ?: $this->getAddressInvoice();
	}

	public function getAddressDeliveryRaw()
	{
		return $this->addressDelivery;
	}

	/**
	 * @return OrderAddress|null
	 */
	public function getAddressInvoice()
	{
		return $this->addressInvoice ?: ($this->addressDelivery ?: null);
	}

	/**
	 * @param Customer $customer
	 *
	 * @return Order
	 */
	public function setCustomer(Customer $customer): Order
	{
		$this->customer = $customer;

		return $this;
	}

	public function setMessage($message)
	{
		$this->message = $message;

		return $this;
	}

	/**
	 * @param OrderPayment $payment
	 *
	 * @return Order
	 */
	public function setPayment($payment)
	{
		$this->payment = $payment;

		return $this;
	}

	/**
	 * @param OrderSpedition $spedition
	 *
	 * @return Order
	 */
	public function setSpedition($spedition)
	{
		$this->spedition = $spedition;

		return $this;
	}

	public function setOrderItems(array $orderItems)
	{
		$this->orderItems = new ArrayCollection($orderItems);

		return $this;
	}

	public function setAddressDelivery(OrderAddress $addressDelivery)
	{
		$this->addressDelivery = $addressDelivery;

		return $this;
	}

	public function setAddressInvoice(OrderAddress $addressInvoice)
	{
		$this->addressInvoice = $addressInvoice;

		return $this;
	}

	/*******
	 * === OrderDiscounts
	 */

	public function getOrderDiscounts()
	{
		return $this->orderDiscounts;
	}

	public function setOrderDiscounts($orderDiscounts): Order
	{
		$this->orderDiscounts = $orderDiscounts;

		return $this;
	}

	public function addOrderDiscount($orderDiscount): Order
	{
		$this->orderDiscounts->add($orderDiscount);

		return $this;
	}

	protected function calculateDiscounts(float $price): float
	{
		foreach ($this->getOrderDiscounts() as $discount) {
			$price += $discount->calculateDiscount($price);
		}

		return $price;
	}

	/*******
	 * === AgreedTerms
	 */

	public function getAgreedTerms()
	{
		return $this->agreedTerms;
	}

	public function setAgreedTerms($agreedTerms): Order
	{
		$this->agreedTerms = $agreedTerms;

		return $this;
	}

	/*******
	 * === OrderFlags
	 */

	public function getOrderFlags()
	{
		return $this->orderFlags;
	}

	public function setOrderFlags($orderFlags): Order
	{
		if (is_array($orderFlags))
			$orderFlags = new ArrayCollection($orderFlags);
		$this->orderFlags = $orderFlags;

		return $this;
	}

	public function addFlag(OrderFlag $flag)
	{
		if (!$this->orderFlags->containsKey($flag->getType())) {
			$this->orderFlags->set($flag->getType(), $flag);
		}
	}

	public function hasFlag($type)
	{
		return ($this->orderFlags->containsKey($type) && $this->orderFlags[$type] == true);
	}

	/*******
	 * === OrderStatuses
	 */

	public function getOrderStatuses()
	{
		return $this->orderStatuses;
	}

	public function getNewestOrderStatus()
	{
		$newestStatus = $this->orderStatuses->last();

		return $newestStatus;
	}

	public function setOrderStatuses($orderStatuses): Order
	{
		$this->orderStatuses = $orderStatuses;

		return $this;
	}

	public function getCreatedTime()
	{
		foreach ($this->orderStatuses as $orderStatus) {
			if ($orderStatus->getStatus()->getId() == 'created') {
				$createdStatus = $orderStatus;
				break;
			}
		}
		if ($createdStatus) {
			return $createdStatus->getCreated();
		}

		return false;
	}

	public function getInvoice(): ?Invoice
	{
		return $this->invoice;
	}

	public function setInvoice(Invoice $invoice)
	{
		$this->invoice = $invoice;
	}

}
