<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Discount;

use Core\Model\UI\BaseControl;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\CartHelper;
use EshopOrders\Model\Entities\OrderDiscount;
use Nette\Neon\Exception;
use Nette\Utils\ArrayHash;
use Nette\Http\Session;
use Nette\Http\SessionSection;

class DiscountForm extends BaseControl
{
	/** @var Carts */
	private $cartsService;
	
	/** @var SessionSection */
	protected $sessionSection;
	
	/** @var CartHelper */
	protected $cartHelperService;
	
	public function __construct(Session $session, Carts $carts, CartHelper $cartHelper)
	{
		$this->sessionSection = $session->getSection('eshopOrdersOrderForm'); //pro pouziti i v objednavce
		
		$this->cartsService      = $carts;
		$this->cartHelperService = $cartHelper;
	}
	
	public function render()
	{
		$this->template->setFile($this->getTemplateFile());
		$this->template->render();
	}
	
	public function attached($presenter)
	{
		parent::attached($presenter);
		
		if($this->sessionSection->discountCode) {
			if($this->cartHelperService->checkValidDiscountCode($this->sessionSection->discountCode)) {
				$this->setDiscountTemplateData($this->sessionSection->discountCode);
			} else {
				unset($this->sessionSection->discountCode);
			}
		}
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();
		//$form->setAjax();

		$form->addText('discountCode', 'eshopOrders.orderPage.discountCode')->setRequired();
		$form->addSubmit('submit', 'eshopOrders.orderPage.discountSubmit')->setHtmlId('discountSubmit');

		$form->onSuccess[] = [$this, 'formOnSuccess'];

		return $form;
	}
	
	public function setDiscountTemplateData($discountCode)
	{
		$type = OrderDiscount::TYPE_PERCENT;
		$cart = $this->cartsService->getCurrentCart();
		$this->template->status       = 'OK';
		$this->template->discountCode = $discountCode;
		$this->template->discountValue = $this->cartHelperService->discountValue;
		$this->template->discountType = $this->cartHelperService->getDiscountTypeText($type);
		$this->template->discountPrice = $this->cartHelperService->countDiscountPrice($cart->getPriceTotal(), $type);
	}

	public function formOnSuccess($form, ArrayHash $values)
	{
		try {
			$cart = $this->cartsService->getCurrentCart();
			$minOrderLimit = $this->cartHelperService->discountMinOrder;
			if($cart->getPriceTotal() < $minOrderLimit) {
				$this->template->status = 'ERR';
				$this->template->message = 'eshopOrders.cart.discountCodeTooCheap';
				$this->template->messageClass = 'eshopOrdersDiscountCheap';
				$this->template->messageParams = ['limit' => $minOrderLimit];
				$this->redrawControl('messages');
				return;
			}
			
			if($this->cartHelperService->checkValidDiscountCode($values->discountCode)) {
				$this->sessionSection->discountCode = $values->discountCode;
				
				$this->template->message = 'eshopOrders.cart.discountCodeAdded';
				$this->template->messageClass = 'eshopOrdersDiscountSuccess';
				$this->setDiscountTemplateData($values->discountCode);
				$this->getParent()->redrawControl('discountForm');
				$this->getParent()->redrawControl('cartDetailTotalPrice');
				$this->getParent()->redrawControl('cartDetailTotalPriceMobile');
				$this->getPresenter()['orderForm']->redrawControl('orderCartDetail1');
				$this->getPresenter()['orderForm']->redrawControl('orderCartDetail2');
			} else {
				$this->template->status = 'ERR';
				$this->template->message = 'eshopOrders.cart.discountCodeInvalid';
				$this->template->messageClass = 'eshopOrdersDiscountInvalid';
				$this->redrawControl('messages');
			}
			
		} catch (\Exception $e) {
			$this->template->status = 'ERR';
			$this->template->message = 'eshopOrders.cart.discountCodeErr';
			$this->template->messageClass = 'eshopOrdersDiscountErr';
			$this->redrawControl('messages');
			return false;
		}
		
	}
}

