<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Order;

use Contributte\EventDispatcher\EventDispatcher;
use Core\Model\Countries;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\PhoneInput;
use EshopOrders\FrontModule\Model\CartHelper;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Dao\Cart;
use EshopOrders\FrontModule\Model\Dao\Spedition;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\Orders;
use EshopOrders\Model\PaymentSpeditions;
use EshopOrders\Model\Statuses;
use Nette\Forms\Controls\Button;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\ArrayHash;
use Users\Model\Http\UserStorage;

class OrderForm extends BaseControl
{
	const PERSONAL_COLUMNS = ['company', 'firstName', 'lastName', 'phone', 'email', 'country', 'street', 'city',
		'postal', 'idNumber', 'vatNumber'];

	/** @var Statuses */
	protected $statusesService;

	/** @var CartHelper */
	protected $cartHelperService;

	/** @var IOrderCartDetailFactory */
	protected $orderCartDetailFactory;

	/** @var IPaySpedSummaryFactory */
	protected $paySpedSummaryFactory;

	/** @var IOrderSummaryFactory */
	protected $orderSummaryFactory;

	/** @var Orders */
	private $ordersService;

	/** @var Carts */
	private $cartsService;

	/** @var Speditions */
	private $speditionsService;

	/** @var Payments */
	private $paymentsService;

	/** @var PaymentSpeditions */
	protected $paymentSpeditionsService;

	/** @var Customers */
	private $customersService;

	/** @var Countries */
	protected $countriesService;

	/** @var SessionSection */
	protected $sessionSection;

	/** @var Spedition[] */
	protected $speditions;

	/** @var Payment[] */
	protected $payments;

	/** @var PaymentSpedition[] */
	private $paymentSpeditions;

	/** @var Cart */
	protected $cart;

	/** @var array */
	protected $orderPageConfig;

	/** @var UserStorage */
	protected $userStorage;

	/** @var Session */
	protected $session;

	public function __construct($data, IOrderCartDetailFactory $orderCartDetailFactory, IPaySpedSummaryFactory $paySpedSummaryFactory, IOrderSummaryFactory $orderSummaryFactory, Orders $orders, Carts $carts, CartHelper $cartHelper, Speditions $speditions, Payments $payments, PaymentSpeditions $paymentSpeditions, Customers $customersService, Statuses $statuses, Countries $countries, Session $session, UserStorage $userStorage)
	{
		$this->ordersService            = $orders;
		$this->cartsService             = $carts;
		$this->cartHelperService        = $cartHelper;
		$this->speditionsService        = $speditions;
		$this->paymentsService          = $payments;
		$this->paymentSpeditionsService = $paymentSpeditions;
		$this->customersService         = $customersService;
		$this->statusesService          = $statuses;
		$this->countriesService         = $countries;
		$this->userStorage              = $userStorage;
		$this->session                  = $session;

		$this->orderCartDetailFactory = $orderCartDetailFactory;
		$this->paySpedSummaryFactory  = $paySpedSummaryFactory;
		$this->orderSummaryFactory    = $orderSummaryFactory;

		$this->sessionSection = $session->getSection('eshopOrdersOrderForm');

		$this->orderPageConfig = $data['orderPage'];

		$this->payments          = $this->paymentsService->getAllPublished();
		$this->paymentSpeditions = $this->paymentSpeditionsService->getAllPublished();

		$this->cart = $this->cartsService->getCurrentCart();
	}

	public function render()
	{
		$this->cart = $this->cartsService->getCurrentCart();
		$this->template->setFile($this->getTemplateFile());

		$this->template->visibleCountSpedition = $this->orderPageConfig['visibleCountSpedition'];
		$this->template->visibleCountPayment   = $this->orderPageConfig['visibleCountPayment'];

		$this->template->speditions        = $this->getSpeditions();
		$this->template->payments          = $this->payments;
		$this->template->paymentSpeditions = $this->paymentSpeditions;
		$this->template->cart              = $this->cart;

		$form                              = $this['form'];
		$this->template->speditionHtmlName = $form['spedition']->getName();
		$this->template->speditionHtmlId   = $form['spedition']->getHtmlId();
		$this->template->speditionVal      = $form['spedition']->getValue();
		$this->template->paymentHtmlName   = $form['payment']->getHtmlName();
		$this->template->paymentHtmlId     = $form['payment']->getHtmlId();
		$this->template->paymentVal        = $form['payment']->getValue();
		$this->template->countries         = $this->countriesService->getAllNameColumn();

		$paySpedStructured = [];
		$paymentSpeditions = $this->paymentSpeditionsService->getAllPublished();
		foreach ($paymentSpeditions as $key => $item) {
			$paySpedStructured[$item->getSpedition()->getId()][] = $item->getPayment()->getId();
			$paySpedReversed[$item->getPayment()->getId()][]     = $item->getSpedition()->getId();
		}

		$this->template->paymentSpeditionsStructured = $paySpedStructured;
		$this->template->paymentSpeditionsReversed   = $paySpedReversed;

		$this->template->orderData = $this->sessionSection->orderFormData;

		// TODO remove
		$discountCoupon = $this->cartHelperService->getDiscountCoupon($this->cart->getPriceTotal());
		if ($discountCoupon) {
			$this->template->discountCoupon = $discountCoupon;
		}

		$this->template->render();
	}

	public function attached($presenter)
	{
		parent::attached($presenter);

		if ($this->sessionSection->orderFormData) {
			$this->setOrderData($this->sessionSection->orderFormData);
		}
	}

	/*******************************************************************************************************************
	 * ======================== Handle
	 */

	public function handleSaveStep2()
	{
		$request = $this->getPresenter()->getHttpRequest();

		foreach (['spedition', 'payment'] as $k) {
			$this->sessionSection->orderFormData[$k] = $request->getPost($k);
		}

		$this->redrawControl('orderCartDetail');
		$this->redrawControl('orderSummary');
	}

	public function handleSaveStep3()
	{
		$request = $this->getPresenter()->getHttpRequest();

		foreach (self::PERSONAL_COLUMNS as $k) {
			$this->sessionSection->orderFormData[$k]       = $request->getPost($k);
			$this->sessionSection->orderFormData[$k . '2'] = $request->getPost($k . '2');
		}

		$countries                                           = $this->countriesService->getAllNameColumn();
		$this->sessionSection->orderFormData['countryText']  = $countries[$request->getPost('country')] ?? null;
		$this->sessionSection->orderFormData['country2Text'] = $countries[$request->getPost('country2')] ?? null;

		foreach (['useAddrDeli'] as $k) {
			$this->sessionSection->orderFormData[$k] = $request->getPost($k);
		}

		$this['form']['validatePersonal']->validate();

		$this->redrawControl('orderSummary');
	}

	/*******************************************************************************************************************
	 * ======================== Setters
	 */

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{

		$form = $this->createForm();
		$form->setAjax();
		$cart      = $this->cartsService->getCurrentCart();
		$cartValue = $cart->getCartItemsPrice();

		$form->addGroup('speditions');
		$paymentChoices = $speditionChoices = [];
		$speditions     = $this->speditionsService->getAllByCartValue($cartValue);
		$payments       = $this->paymentsService->getAllByCartValue($cartValue);
		foreach ($speditions as $s) {
			$speditionChoices[$s->getId()] = $s->getName() . ' - ' . $s->getPriceActual($cart) . ' Kč'; //TODO cuurency
		}
		foreach ($payments as $p) {
			$paymentChoices[$p->getId()] = $p->getName() . ' - ' . $p->getPrice() . ' Kč'; //TODO cuurency
		}
		$form->addRadioList('spedition', 'eshopOrders.orderPage.spedition', $speditionChoices)->setRequired();
		$form->addRadioList('payment', 'eshopOrders.orderPage.payment', $paymentChoices)->setRequired();

		$form->addSubmit('validateSpedition', 'eshopOrders.orderPage.nextStep')
			->setValidationScope([
				$form['payment'],
				$form['spedition'],
			])->onClick[] = [$this, 'handleSaveStep2'];

		$form->addGroup('personal');
		$form->addText('firstName', 'eshopOrders.orderPage.name')->setRequired();
		$form->addText('lastName', 'eshopOrders.orderPage.lastName')->setRequired();
		$form->addPhone('phone', 'eshopOrders.orderPage.phone')
			->addRule(PhoneInput::PHONE, PhoneInput::$phoneMessages)->setRequired();
		$form->addEmail('email', 'eshopOrders.orderPage.email')->setRequired();
		$form->addText('street', 'eshopOrders.orderPage.street')->setRequired();
		$form->addText('city', 'eshopOrders.orderPage.city')->setRequired();
		$form->addText('postal', 'eshopOrders.orderPage.postal')->setRequired();
		$countries = $this->countriesService->getAllNameColumn();
		$form->addSelect('country', 'eshopOrders.orderPage.country', $countries);
		$form->addCheckbox('companyOrder', 'eshopOrdersForm.orderPage.companyOrder');
		$form->addText('company', 'eshopOrders.orderPage.company')
			->addConditionOn($form['companyOrder'], BaseForm::EQUAL, true)->setRequired();
		$form->addText('idNumber', 'eshopOrders.orderPage.idNumber')
			->addConditionOn($form['companyOrder'], BaseForm::EQUAL, true)->setRequired();
		$form->addText('vatNumber', 'eshopOrders.orderPage.vatNumber');

		//fakturacni adresa
		$useAddrDeli = $form->addCheckbox('useAddrDeli', 'eshopOrders.orderPage.useAddrDeli');
		$form->addText('company2', 'eshopOrders.orderPage.company');
		$form->addText('firstName2', 'eshopOrders.orderPage.name')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)->setRequired();
		$form->addText('lastName2', 'eshopOrders.orderPage.lastName')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)->setRequired();
		$form->addEmail('email2', 'eshopOrders.orderPage.email')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)->setRequired();
		$form->addPhone('phone2', 'eshopOrders.orderPage.phone')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)
			->addRule(PhoneInput::PHONE, PhoneInput::$phoneMessages)->setRequired();
		$form->addText('street2', 'eshopOrders.orderPage.street')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)->setRequired();
		$form->addText('city2', 'eshopOrders.orderPage.city')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)->setRequired();
		$form->addText('postal2', 'eshopOrders.orderPage.postal')
			->addConditionOn($useAddrDeli, BaseForm::EQUAL, true)->setRequired();
		$form->addSelect('country2', 'eshopOrders.orderPage.country', $countries);

		$validateFields = [];
		foreach (self::PERSONAL_COLUMNS as $v) {
			$validateFields[] = $form[$v];

			if (!in_array($v, ['idNumber', 'vatNumber']))
				$validateFields[] = $form[$v . '2'];
		}

		$form->addSubmit('validatePersonal', 'eshopOrders.orderPage.nextStep')
			->setValidationScope($validateFields)
			->onClick[] = [$this, 'handleSaveStep3'];

		$form->addTextArea('messageAlt', 'eshopOrders.orderPage.message')->setAttribute('class', ['messageInput']);
		$form->addTextArea('message', 'eshopOrders.orderPage.message');

		$form->addCheckbox('agreedTerms', 'eshopOrders.orderPage.agreedTerms')->setRequired();
		$form->addCheckbox('agreedQuestioning', 'eshopOrders.orderPage.agreedQuestioning')->setDefaultValue(1);

		//tlacitko pro ulozeni dat do session mezi kroky objednavky
		$saveButtonCallback = [$this, 'saveFormData'];
		$form->addSubmit('saveData', 'save')
			->setValidationScope(false)
			->setHtmlAttribute('class', ['saveFormData'])
			->onClick[]     = function(Button $button) use ($saveButtonCallback) {
			$values = $button->getForm()->getValues(true);
			$saveButtonCallback($values);
		};

		$form->addSubmit('submit', 'eshopOrders.orderPage.send')
			->setAttribute('class', ['submitButton', 'eshopOrdersNext4'])
			->setHtmlId('orderFormSubmit');

		if ($this->userStorage->getIdentity()) {
			$customer = $this->customersService->getByUser($this->userStorage->getIdentity());

			if ($customer) {
				$addrDeli = $customer->getAddressDelivery();
				$addrInvo = $customer->getAddressInvoice();

				if ($addrDeli)
					$form->setDefaults($addrDeli->toArray());
				if ($addrInvo) {
					$def = [];
					foreach ($addrInvo->toArray() as $k => $v)
						$def[$k . '2'] = $v;
					$form->setDefaults($def);
				}
			}
		}

		$form->onValidate[] = [$this, 'formOnValidate'];
		$form->onSuccess[]  = [$this, 'formOnSuccess'];

		$orderFormEvent           = new OrderFormEvent($form);
		$orderFormEvent->template = $this->template;
		$this->eventDispatcher->dispatch('eshopOrders.createOrderForm', $orderFormEvent);

		return $form;
	}

	protected function saveFormData($data)
	{
		$cart       = $this->cartsService->getCurrentCart();
		$cartValue  = $cart->getCartItemsPrice();
		$priceTotal = $cartValue;
		if ($data['spedition'])
			$priceTotal += $this->getSpeditions()[$data['spedition']]->getPriceActual($cart);
		if ($data['payment'])
			$priceTotal += $this->payments[$data['payment']]->getPrice();
		if ($this->sessionSection->discountCode) {
			$type       = OrderDiscount::TYPE_PERCENT;
			$priceTotal += $this->cartHelperService->countDiscountPrice($cartValue, $type);
		}
		$data['priceTotal'] = $priceTotal;

		if ($data['message']) {
			$data['messageAlt'] = $data['message'];
		} else {
			$data['message'] = $data['messageAlt'];
		}

		$this->eventDispatcher->dispatch('eshopOrders.saveOrderFormData', new SaveOrderFormDataEvent($data, $cart));

		$this->sessionSection->orderFormData = $data;

		$this->redrawControl('orderCartDetailSummary');
		$this->redrawControl('orderSummary');
		$this->redrawControl('paySpedSummary');
	}

	public function setOrderData(array $orderData): void
	{
		$this->eventDispatcher->dispatch('eshopOrders.setOrderFormData', new SetOrderFormDataEvent($orderData, $this['form']));

		$d = [
			'spedition'         => $orderData['spedition'],
			'payment'           => $orderData['payment'],
			'messageAlt'        => $orderData['messageAlt'],
			'message'           => $orderData['message'],
			'useAddrDeli'       => $orderData['useAddrDeli'],
			'agreedTerms'       => $orderData['agreedTerms'],
			'agreedQuestioning' => $orderData['agreedQuestioning'],
		];
		foreach (self::PERSONAL_COLUMNS as $k) {
			$d[$k] = $orderData[$k];

			if (!in_array($k, ['idNumber', 'vatNumber']))
				$d[$k . '2'] = $orderData[$k . '2'];
		}

		$this['form']->setDefaults($d);
	}

	public function formOnValidate(BaseForm $form): bool
	{
		$cart                                = $this->cartsService->getCurrentCart();
		$cartItems                           = $cart->getCartItems();
		$this->template->formSummaryMessages = [];

		if (empty($cartItems)) {
			$form->addError('eshopOrders.orderPage.emptyCart');
			$this->template->formSummaryMessages = ['eshopOrders.orderPage.emptyCartData'];
		}

		$this->eventDispatcher->dispatch('eshopOrders.orderFormValidate', new OrderFormEvent($form));

		if ($form->hasErrors()) {
			$errs = [];
			foreach ($form->getComponents(true) as $c) {
				if ($c->getErrors())
					$errs[$c->getHtmlId()] = $c->getErrors();
			}

			$this->getPresenter()->payload->orderFormErrors = $errs;

			foreach ($form->getErrors() as $e)
				$this->template->formSummaryMessages[] = $e;
			$this->redrawControl('formMessages');

			return false;
		} else {
			$this->getPresenter()->payload->orderFormIsValid = true;
		}

		return true;
		// nebude potreba take validovat jestli doprava a platba jsou aktivni a plati pro cenu kosiku?
	}

	public function formOnSuccess(BaseForm $form, ArrayHash $values)
	{
		if (!$form['submit']->isSubmittedBy()) {
			return;
		}

		try {
			$ident = $this->genUuid();
			$order = new Order($ident);

			$order->setMessage($values->message ?: $values->messageAlt);
			$order->setAgreedTerms($values->agreedTerms);

			if ($values->agreedQuestioning) {
				$orderFlagQuestioning = new OrderFlag(OrderFlag::TYPE_QUESTIONING, true, $order);
				$order->addFlag($orderFlagQuestioning);
			}

			$addrInv = new OrderAddress($this->ordersService::ADDRESS_INVOICE);
			$addrInv->setFirstName($values->firstName);
			$addrInv->setLastName($values->lastName);
			$addrInv->setEmail($values->email);
			$addrInv->setPhone($values->phone);
			$addrInv->setStreet($values->street);
			$addrInv->setCity($values->city);
			$addrInv->setPostal($values->postal);
			if ($values->country) {
				$addrInv->setCountry($this->countriesService->getReference($values->country));
			}

			if ($values->companyOrder) {
				$addrInv->setCompany($values->company);
				$addrInv->setIdNumber($values->idNumber);
				$addrInv->setVatNumber($values->vatNumber);
			}
			$this->em->persist($addrInv);

			$order->setAddressInvoice($addrInv);

			if ($values->useAddrDeli) {
				$addrDeli = new OrderAddress($this->ordersService::ADDRESS_DELIVERY);
				$addrDeli->setFirstName($values->firstName2);
				$addrDeli->setLastName($values->lastName2);
				$addrDeli->setEmail($values->email2);
				$addrDeli->setPhone($values->phone2);
				$addrDeli->setStreet($values->street2);
				$addrDeli->setCity($values->city2);
				$addrDeli->setPostal($values->postal2);
				if ($values->country2)
					$addrDeli->setCountry($this->countriesService->getReference($values->country2));
				$addrDeli->setCompany($values->company2);
				$this->em->persist($addrDeli);

				$order->setAddressDelivery($addrDeli);
			}

			$customer = $this->getOrCreateCustomer(
				$order->getAddressInvoice()->getEmail(),
				$order->getAddressInvoice()->getFirstName(),
				$order->getAddressInvoice()->getLastName(),
				$order->getAddressInvoice()->getPhone());

			if ($customer) {
				$order->setCustomer($customer);
				$customerDeli = $customer->getAddressDelivery();
				$customerInvo = $customer->getAddressInvoice();

				if ($addrInv) {
					if (!$customerInvo)
						$customerInvo = new CustomerAddress($customer);
					$customerInvo->fillFromOrderAddress($addrInv);
					$this->em->persist($customerInvo);
				}

				if ($addrDeli) {
					if (!$customerDeli)
						$customerDeli = new CustomerAddress($customer);
					$customerDeli->fillFromOrderAddress($addrDeli);
					$this->em->persist($customerDeli);
				}
			}

			$cart       = $this->cartsService->getCurrentCart();
			$cartItems  = $cart->getCartItems();
			$orderItems = $this->ordersService->fillOrderItems($cartItems, $order);
			$order->setOrderItems($orderItems);

			$orderPayment = new OrderPayment($this->paymentsService->getRawReference($values->payment), $order);
			$order->setPayment($orderPayment);

			$orderSpedition = new OrderSpedition($this->speditionsService->getRawReference($values->spedition), $order);
			$spedition      = $this->speditionsService->get($values->spedition);
			$orderSpedition->setPrice($spedition->getPriceActual($cart)); //pripadne doprava zdarma podle ceny objednavky nebo podle stitku produktu
			$order->setSpedition($orderSpedition);

			$statusCreated      = $this->statusesService->get('created');
			$orderActionCreated = new OrderStatus($order, $statusCreated);

			foreach ($cart->getDiscounts() as $k => $v) {
				$discount = new OrderDiscount('', $order);
				$discount->setValue($v->amount);
				$discount->setType($v->type);
				$discount->setPrice($v->calculateDiscount($cart->getCartItemsPrice()));
				$discount->setName($this->translator->translate('eshopOrders.discount') . ': ' . $v->amount . $v->typeSymbol);
				$this->em->persist($discount);
				$order->addOrderDiscount($discount);
			}

			//ale mozna ze to nebude $values->discountCode, ale session[discountCode]
			if ($this->sessionSection->discountCode && $this->cartHelperService->checkValidDiscountCode($this->sessionSection->discountCode)) {
				$type     = OrderDiscount::TYPE_PERCENT;
				$typeText = $this->cartHelperService->getDiscountTypeText($type);
				$discount = new OrderDiscount($this->sessionSection->discountCode, $order);
				$discount->setValue($this->cartHelperService->discountValue);
				$discount->setType($type);
				$discount->setPrice($this->cartHelperService->countDiscountPrice($cart->getCartItemsPrice(), $type));
				$discount->setName($this->translator->translate('eshopOrders.orderPage.discount') . $discount->getValue() . $typeText);
				$order->addOrderDiscount($discount); //nutne, jinak se objednavka zacachuje bez slevovych kodu a v emailu budou chybet
				unset($this->sessionSection->discountCode); //smazeme vlozeny kod ze session -> v nove objednavce vlozit znova

				//TODO promítnout do celkové ceny objednávky
			}

			$this->em->getConnection()->beginTransaction();
			try {
				$eventOrder                 = new OrderEvent($order);
				$eventOrder->addrDelivery   = $addrDeli;
				$eventOrder->addrInvoice    = $addrInv;
				$eventOrder->orderSpedition = $orderSpedition;
				$eventOrder->orderPayment   = $orderPayment;
				$eventOrder->orderItems     = $orderItems;
				$eventOrder->formData       = $values;
				$this->eventDispatcher->dispatch('eshopOrders.orderBeforeSave', $eventOrder);
				//ulozit objednavku
				$this->em->persist($order);
				//ulozit k ni adresy
				$this->em->persist($addrInv);
				if ($addrDeli)
					$this->em->persist($addrDeli);
				//polozky doprava a platba
				$this->em->persist($orderSpedition);
				$this->em->persist($orderPayment);
				//ulozit k ni polozky
				foreach ($orderItems as $oi) {
					$this->em->persist($oi);
				}
				//akce - datum kdy byla objednavka vytvorena
				$this->em->persist($orderActionCreated);
				//pripadny slevovy kod
				if ($discount)
					$this->em->persist($discount);
				//pripadne nastavene priznaky
				if ($orderFlagQuestioning)
					$this->em->persist($orderFlagQuestioning);
				//teprve po ulozeni vseho flush (aby byly spravne propojene cizi klice)
				$this->em->flush();
				$this->em->getConnection()->commit();

				//TODO nejake promazavani kosiku z DB, kdyz uz k nim neni aktivni session... nebo po nejakem timeoutu (tyden a pod)
			} catch (Exception $e) {
				$this->em->getConnection()->rollBack();
				throw $e;
			}

			//dale uz pracujeme s objednavkou nactenou z DB
			$orderDetail = $this->ordersService->get($order->getId());
			unset($order);
			$this->cartsService->deleteCurrentCart();
			$this->sessionSection->orderFormData = [];

			// Spuštění události po dokončení objednávky jako odeslání emailu, zboží,...
			$this->eventDispatcher->dispatch('eshopOrders.orderOnSuccess', new OrderEvent($orderDetail));
			// Vytáhnutí zpráv z událostí
			$messages = $this->session->getSection('eshopOrders/orderSuccessMessages');

			if ($messages->success)
				foreach ($messages->success as $v)
					$this->getPresenter()->flashMessageSuccess($v);
			if ($messages->error)
				foreach ($messages->error as $v)
					$this->getPresenter()->flashMessageDanger($v);
			$messages->remove();
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		$this->getPresenter()->redirect('Finished:orderFinished', ['orderIdent' => $orderDetail->getIdent()]);

		return true;
	}

	protected function getOrCreateCustomer($email, $firstName = '', $lastName = '', $phone = '')
	{
		$userId = null;
		if ($this->getPresenter()->getUser()->isLoggedIn()) {
			$userId = $this->getPresenter()->getUser()->id;
		}
		$customer = $this->customersService->getOrCreateCustomer($userId, $email, $firstName, $lastName, $phone);

		return $customer;
	}

	/** http://stackoverflow.com/a/2040279
	 * @return string UUID - unikatni ID, tezke na uhodnuti
	 */
	protected function genUuid()
	{
		return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
			// 32 bits for "time_low"
			mt_rand(0, 0xffff), mt_rand(0, 0xffff),

			// 16 bits for "time_mid"
			mt_rand(0, 0xffff),

			// 16 bits for "time_hi_and_version",
			// four most significant bits holds version number 4
			mt_rand(0, 0x0fff) | 0x4000,

			// 16 bits, 8 bits for "clk_seq_hi_res",
			// 8 bits for "clk_seq_low",
			// two most significant bits holds zero and one for variant DCE1.1
			mt_rand(0, 0x3fff) | 0x8000,

			// 48 bits for "node"
			mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
		);
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrderCartDetail()
	{
		$control = $this->orderCartDetailFactory->create();

		return $control;
	}

	protected function createComponentPaySpedSummary()
	{
		$control = $this->paySpedSummaryFactory->create();

		$orderData = $this->sessionSection->orderFormData;
		$control->setParameters($orderData['spedition'], $orderData['payment'], $orderData['priceTotal']);

		return $control;
	}

	protected function createComponentOrderSummary()
	{
		$control = $this->orderSummaryFactory->create();

		$orderData = $this->sessionSection->orderFormData;
		$control->setParameters($orderData);

		return $control;
	}

	public function getSpeditions()
	{
		if ($this->speditions === null) {
			$this->speditions = $this->speditionsService->getAllPublished();
		}

		return $this->speditions;
	}
}
