<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use EshopOrders\Model\Orders;

class OrdersGrid extends BaseControl
{
	/** @var Orders */
	protected $ordersService;

	/** @var array */
	protected $config;

	/** @var array */
	protected $exportCallbacks;

	public function __construct($data, Orders $orders)
	{
		$this->ordersService = $orders;
		$this->config        = $data;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();

		$qb = $this->ordersService->getEr()->createQueryBuilder('o')
			->addSelect('oi, ad')
			->leftJoin('o.addressInvoice', 'oi')
			->leftJoin('o.addressDelivery', 'ad')
			->orderBy('o.id', 'DESC');
		$grid->setDataSource($qb);

		//Columns
		$grid->addColumnLink('id', 'eshopOrders.default.orderNumber', 'Default:editOrder')->setAlign('right');
		$grid->addColumnText('firstName', 'default.name', 'addressDelivery.firstName');
		$grid->addColumnText('lastName', 'eshopOrders.default.lastName', 'addressDelivery.lastName');
		$grid->addColumnText('city', 'eshopOrders.default.city', 'addressDelivery.city');
		$grid->addColumnText('spedition', 'eshopOrders.default.spedition', 'spedition.name');
		$grid->addColumnText('payment', 'eshopOrders.default.payment', 'payment.name');
		$grid->addColumnDateTime('created', 'eshopOrders.defaultGrid.createdTime', 'createdTime')->setFormat('d.m.Y H:i');
		$grid->addColumnText('status', 'eshopOrders.default.status', 'newestOrderStatus.status.name');
		$grid->addColumnNumber('price', 'eshopOrders.default.price')->setFormat(2, ',', ' ');

		// Filter

		// ItemDetail
		$grid->setItemsDetail($this->getTemplateFile('OrdersGridDetail', 'Order'));

		// Actions
		$grid->addAction('edit', '', 'Default:editOrder')->setIcon('edit')->setBsType('primary');
		if ($this->config['allowOrderDelete']) {
			$grid->addAction('delete', '', 'delete!')->setConfirm('eshopOrders.orderForm.reallyDelete')->setIcon('times')->setBsType('danger');
		}

		// Group action
		if ($this->exportCallbacks) {
			foreach ($this->exportCallbacks as $v) {
				$grid->addGroupAction($v['title'])->onSelect[] = $v['callback'];
			}
		}

		// Columns prototype
		$grid->getColumn('id')->getElementPrototype('th')->class[] = 'w1nw';

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->config['allowOrderDelete']) {
			if ($this->ordersService->remove($id))
				$presenter->flashMessageSuccess('default.removed');
			else
				$presenter->flashMessageDanger('default.removeFailed');
		} else {
			$presenter->flashMessageDanger('eshopOrders.orderForm.removeDenied');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */


	/*******************************************************************************************************************
	 * =================  GET SET
	 */

	public function addExportCallback(string $key, string $title, callable $callback): self
	{
		$this->exportCallbacks[$key] = [
			'title'    => $title,
			'callback' => $callback,
		];

		return $this;
	}
}
