<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Customer;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\Model\Entities\GroupCustomers;
use Nette\Application\UI\Form;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\AdminModule\Model\Customers;
use Users\Model\Entities\User;

class CustomerForm extends BaseControl
{
	/** @var Customer */
	public $customer;

	/** @var Customers */
	protected $customersService;

	public function __construct(Customers $customers)
	{
		$this->customersService = $customers;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$groups = ['' => ''];
		foreach ($this->em->getRepository(GroupCustomers::class)->findPairs([], 'name', ['name'], 'id') as $k => $v)
			$groups[$k] = $v;

		$form->addText('name', 'default.firstName', 'user.name')->setRequired()->setMaxLength(255);
		$form->addText('lastname', 'default.lastName', 'user.lastname')->setRequired()->setMaxLength(255);
		$form->addText('email', 'default.email')->setRequired()->setMaxLength(255);
		$form->addText('phone', 'default.phone')->setMaxLength(255);
		$form->addSelect('group', 'eshopOrders.customerForm.group', $groups);

		//$form->addText('street', 'default.street')->setRequired()->setMaxLength(255);
		//$form->addText('city', 'default.city')->setRequired()->setMaxLength(255);
		//$form->addText('postal', 'default.zipNumber')->setRequired()->setMaxLength(255);

		$form->addBool('isActive', 'default.isActive', 'user.isActive')->setRequired();
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if (!$this->customer) {
				$this->getPresenter()->flashMessageWarning('eshopOrders.customerForm.noCreate');

				return false;
			}

			$customer     = $this->customer;
			$user         = $customer->getUser();
			$flashMessage = 'eshopOrders.customerForm.edited';

			$customer->setPhone($values->phone);
			$user->setName($values->name);
			$user->setLastname($values->lastname);
			$user->setEmail($values->email);
			if ($values->group) {
				$customer->setGroupCustomers($this->em->getReference(GroupCustomers::class, $values->group));
			} else {
				$customer->removeGroupCustomers();
			}

			if ($this->customersService->isAdmin($customer->getId()) && $values->isActive != $user->isActive()) {
				$this->getPresenter()->flashMessageWarning('eshopOrders.customerForm.noAdminDeactivate');
			} else {
				$user->disable(!$values->isActive);
			}

			$this->em->persist($customer);
			$this->em->persist($user);

			$this->em->flush();
			$form->addCustomData('customerId', $customer->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setCustomer($id)
	{
		$this->customer = $this->customersService->get($id);

		if ($this->customer) {
			$o = $this->customer;
			$u = $o->getUser();
			$this['form']->setDefaults([
				'phone'    => $o->getPhone(),
				'name'     => $u->getName(),
				'lastname' => $u->getLastname(),
				'email'    => $u->getEmail(),
				'isActive' => $u->isActive(),
			]);

			if ($o->getGroupCustomers() && array_key_exists($o->getGroupCustomers()->getId(), $this['form']['group']->getItems()))
				$this['form']['group']->setDefaultValue($o->getGroupCustomers()->getId());
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
