<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;

/**
 * @ORM\Table("eshop_orders__order_item")
 * @ORM\Entity
 */
class OrderItem
{
	use Identifier;
	use MagicAccessors;

	/**
	 * @var int
	 * @ORM\Column(name="product_id", type="integer", nullable=true)
	 */
	protected $productId;
	
	/**
	 * @var int
	 * @ORM\Column(name="variant_id", type="integer", nullable=true)
	 */
	protected $variantId;
	
	/**
	 * @var string
	 * @ORM\Column(name="code1", type="string", length=60, nullable=true)
	 */
	protected $code1;

	/**
	 * @var int
	 * @ORM\Column(name="quantity", type="smallint")
	 */
	protected $quantity;
	
	/**
	 * @var float
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;
	
	/**
	 * @var int
	 * @ORM\Column(name="vat_rate", type="smallint", nullable=true)
	 */
	public $vatRate;

	/**
	 * @var Order
	 * @ORM\ManyToOne(targetEntity="Order", inversedBy="orderItems")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $order;
	
	/**
	 * @var OrderItemTexts[]
	 * @ORM\OneToMany(targetEntity="OrderItemTexts", mappedBy="id", indexBy="lang", cascade={"all"})
	 * TODO moznost optimalizace - nekaskadovat
	 */
	protected $orderItemTexts;

	/**
	 * @var array
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected $moreData = [];

	public function __construct($productId, $order)
	{
		$this->productId      = $productId;
		$this->quantity       = 1;
		$this->order          = $order;
		$this->orderItemTexts = new ArrayCollection();
		$this->moreData          = [];
	}

	/*****
	 * === Quantity
	 */

	public function setQuantity($quantity)
	{
		$this->quantity = $quantity && $quantity > 0 ? $quantity : null;
	}

	public function getQuantity() { return $this->quantity; }
	
	/*******
	 * === Product
	 */
	
	public function getProductId()
	{
		return $this->productId;
	}

	public function setProductId($productId)
	{
		$this->productId = $productId;
		return $this;
	}
	
	/*******
	 * === Variant
	 */
	
	public function getVariantId()
	{
		return $this->variantId;
	}
	
	public function setVariantId($variantId)
	{
		$this->variantId = $variantId;
		return $this;
	}
	
	/*******
	 * === Code1
	 */
	
	public function getCode1()
	{
		return $this->code1;
	}
	
	public function setCode1($code1): OrderItem
	{
		$this->code1 = $code1;
		return $this;
	}
	
	/*******
	 * === Order
	 */
	
	public function getOrder(): Order
	{
		return $this->order;
	}

	public function setOrder(Order $order)
	{
		$this->order = $order;
		return $this;
	}
	
	/*******
	 * === Price
	 */
	
	public function getPrice(): float
	{
		return (float)$this->price;
	}
	
	public function getPriceTotal(): float
	{
		return (float)($this->price * $this->quantity);
	}
	
	public function setPrice(float $price)
	{
		$this->price = $price;
		return $this;
	}
	
	/*******
	 * === VatRate
	 */
	
	public function getVatRate()
	{
		return $this->vatRate;
	}
	
	public function setVatRate($vatRate): OrderItem
	{
		$this->vatRate = $vatRate;
		return $this;
	}
	
	/*******
	 * === OrderItemText
	 */
	
	public function addOrderItemText($lang)
	{
		$this->orderItemTexts->set($lang, new OrderItemTexts($this, $lang));
	}
	
	public function setOrderItemText(OrderItemTexts $orderItemTexts)
	{
		$this->orderItemTexts->set($orderItemTexts->getLang(), $orderItemTexts);
	}
	
	/**
	 * @param string $lang
	 * @return OrderItemTexts
	 */
	public function getOrderItemText($lang)
	{
		return $this->orderItemTexts->get($lang);
	}


	/*************************************
	 * == MoreData
	 */

	/**
	 * @param string $key
	 *
	 * @return mixed|null
	 */
	public function getMoreDataValue($key) { return $this->moreData[$key] ?? null; }

	/**
	 * @param string $key
	 * @param string $value
	 *
	 * @return $this
	 */
	public function setMoreDataValue($key, $value)
	{
		if (!is_array($this->moreData))
			$this->moreData = [];

		$this->moreData[$key] = $value;

		return $this;
	}

	/**
	 * @param array $data
	 *
	 * @return $this
	 */
	public function setMoreData($data)
	{
		$this->moreData = $data;

		return $this;
	}
}

