<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Subscribers;

use Contributte\EventDispatcher\EventSubscriber;
use Core\Model\Mailing\TemplateFactory;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Pdf\IInvoicePdfBuilderFactory;
use Exception;
use Mpdf\Output\Destination;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Localization\ITranslator;
use Nette\Mail\IMailer;
use Nette\Mail\Message;

class OrderMailerSubscriber implements EventSubscriber
{
	/** @var  array */
	protected $conf;

	/** @var ITranslator */
	protected $translator;

	/** @var TemplateFactory */
	protected $templateFactory;

	/** @var IMailer */
	protected $mailer;

	/** @var SessionSection */
	protected $orderSuccessMessages;

	/** @var Invoices */
	protected $invoices;

	/** @var IInvoicePdfBuilderFactory */
	protected $invoicePdfBuilderFactory;

	public function __construct(array $conf, ITranslator $translator, TemplateFactory $templateFactory, IMailer $mailer, Session $session, Invoices $invoices, IInvoicePdfBuilderFactory $invoicePdfBuilderFactory)
	{
		$this->conf                 	= $conf;
		$this->translator           	= $translator;
		$this->templateFactory      	= $templateFactory;
		$this->mailer               	= $mailer;
		$this->orderSuccessMessages 	= $session->getSection('eshopOrders/orderSuccessMessages');
		$this->invoices					= $invoices;
		$this->invoicePdfBuilderFactory = $invoicePdfBuilderFactory;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderOnSuccess' => ['orderOnSuccess', 100],
		];
	}

	/**
	 * @param OrderEvent $event
	 * @throws Exception
	 */
	public function orderOnSuccess(OrderEvent $event): void
	{
		if ($this->conf['send'] == false)
			return;

		$order      = $event->order;
		$conf       = $this->conf;
		$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/orderSend.latte';
		$originFile = __DIR__ . '/../../_emails/orderSend.latte';

		if (!file_exists($file))
			$file = $originFile;

		if (!$conf['fromEmail'] || !$conf['toEmail'])
			return;

		$template = $this->templateFactory->create();

		$subject = $this->translator->translate('eshopOrders.emails.subject', [
			'orderId'  => $order->getId(),
			'siteName' => $template->siteName,
		]);

		$template->setFile($file);
		$template->setParameters([
			'subject'        => $subject,
			'order'          => $order,
			'originTemplate' => $originFile,
		]);

		$message = new Message();
		$message->setSubject($subject);
		$message->setFrom($conf['fromEmail'], $conf['fromName'] ?? null);
		$message->addTo($conf['toEmail'], $conf['toName'] ?? null);

		if (isset($conf['bcc']) && $conf['bcc']) {
			foreach (explode(',', $conf['bcc']) as $v) {
				$message->addBcc(trim($v));
			}
		}

		$message->setHtmlBody($template->renderToString());

		$message->addTo($order->getAddressInvoice()->getEmail(), $order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName());

		$invoice = $this->invoices->getOneByOrder($event->order->getId());

		if ($invoice !== null && EshopOrdersConfig::load('invoice.sendWithOrder')) {
			$invoice->order = $order;
			$pdfBuilder = $this->invoicePdfBuilderFactory->create($invoice);
			$message->addAttachment($pdfBuilder->getFileName(), $pdfBuilder->render(Destination::STRING_RETURN));
		}

		$this->mailer->send($message);
	}
}
