<?php declare(strict_types=1);

namespace EshopOrders\Model\Pdf;

use Core\Model\Pdf\Pdf;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\InvoiceConfigRepository;
use EshopOrders\Model\Invoices;
use Mpdf\Mpdf;
use Mpdf\MpdfException;
use Mpdf\Output\Destination;
use Nette\DI\Container;
use Nette\Localization\ITranslator;
use Nette\Utils\Strings;

class InvoicePdfBuilder
{
	/** @var Invoice */
	protected $invoice;

	/** @var InvoiceConfigRepository */
	protected $invoiceConfigRepository;

	/** @var Invoices */
	protected $invoices;

	/** @var ITranslator */
	protected $translator;

	/** @var Pdf */
	protected $pdfService;

	/** @var Container */
	protected $container;

	/**
	 * InvoicePdfBuilder constructor.
	 * @param Invoice $invoice
	 * @param InvoiceConfigRepository $invoiceConfigRepository
	 * @param Invoices $invoices
	 * @param ITranslator $translator
	 * @param Pdf $pdfService
	 * @param Container $container
	 */
	public function __construct(Invoice $invoice, InvoiceConfigRepository $invoiceConfigRepository, Invoices $invoices, ITranslator $translator, Pdf $pdfService, Container $container)
	{
		$this->invoice = $invoice;
		$this->invoiceConfigRepository = $invoiceConfigRepository;
		$this->invoices = $invoices;
		$this->translator = $translator;
		$this->pdfService = $pdfService;
		$this->container = $container;
	}

	/**
	 * @param string $destination
	 * @return string
	 * @throws MpdfException
	 */
	public function render(string $destination = Destination::INLINE): string
	{
		$pdf = $this->build();
		return $pdf->Output($this->getFileName(), $destination);
	}

	/**
	 * @return Mpdf
	 * @throws MpdfException
	 */
	private function build(): Mpdf
	{
		// css
		$customCssFile = TEMPLATES_DIR . '/Front/default/EshopOrders/_pdf/invoiceStyle.css';
		$defaultCssFile = SRC_DIR . '/eshoporders/src/Templates/_pdf/invoiceStyle.css';

		// latte
		$customTemplateFile = TEMPLATES_DIR . '/Front/default/EshopOrders/_pdf/invoice.latte';
		$defaultTemplateFile = SRC_DIR . '/eshoporders/src/Templates/_pdf/invoice.latte';

		// build template
		$template = $this->pdfService->getTemplateFactory();
		$template->setFile(file_exists($customTemplateFile) ? $customTemplateFile : $defaultTemplateFile);
		$template->invoice = $this->invoice;
		$template->invoiceData = $this->invoice->invoiceData;
		$template->customer = $this->invoice->invoiceData->getCustomer();
		$template->supplier = $this->invoice->invoiceData->getSupplier();
		$template->bank = $this->invoice->invoiceData->getSupplier()->getBank();
		$template->isPayerVat = $this->invoice->invoiceData->getSupplier()->isPayerVat;
		$template->container = $this->container;

		// service config
		$pdf = $this->pdfService->newPdf();
		$pdf->WriteHTML(file_get_contents(file_exists($customCssFile) ? $customCssFile : $defaultCssFile), 1);
		$pdf->WriteHTML($template->renderToString(), 0);
		$pdf->SetTitle($this->getTitle());

		return $pdf;
	}

	/**
	 * @return string
	 */
	private function getTitle(): string
	{
		return $this->translator->translate('eshopOrders.title.invoice', null, ['id' => $this->invoice->ident]);
	}

	/**
	 * @return string
	 */
	public function getFileName(): string
	{
		return sprintf('%s.pdf', Strings::webalize($this->getTitle()));
	}

}