<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\FrontModule\Model\Dao;

/**
 * TODO netahat celé entity z db
 *
 * Class Speditions
 * @package EshopOrders\FrontModule\Model
 */
class Speditions extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = Spedition::class;

	/**
	 * @param int $id
	 *
	 * @return Dao\Spedition|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function get($id)
	{
		$speditionRaw = $this->loadRawSpedition($id);
		if (!$speditionRaw) {
			return null;
		}
		$spedition = $this->fillDao([$speditionRaw])[$id];

		return $spedition;
	}

	/** Pozor - vraci dopravy nezavisle na hodnote kosiku, takze i ty ktere nemusi byt platne
	 * @return Dao\Spedition[] vsechny aktivni dopravy
	 */
	public function getAllPublished()
	{
		$speditionsRawQuery = $this->getEr()->createQueryBuilder('s', 's.id');
		$speditionsRawQuery->andWhere('s.isPublished = :isPublished')->setParameter('isPublished', true);
		$speditionsRawQuery->orderBy('s.position');
		$speditionsRaw = $speditionsRawQuery->getQuery()->getResult();

		$speditions = $this->fillDao($speditionsRaw);

		return $speditions;
	}

	/**
	 * @param int $cartValue cena objednavky
	 *
	 * @return Dao\Spedition[] vsechny dopravy, ktere jsou aktivni, a jsou dostupne pro danou cenu objednavky
	 */
	public function getAllByCartValue($cartValue)
	{
		$speditionsRawQuery = $this->getEr()->createQueryBuilder('s', 's.id');
		$speditionsRawQuery->andWhere('s.isPublished = :isPublished')->setParameter('isPublished', true);
		$speditionsRawQuery->andWhere('s.availableFrom IS NULL OR s.availableFrom <= :cartValue')->setParameter('cartValue', $cartValue);
		$speditionsRawQuery->andWhere('s.availableTo IS NULL OR s.availableTo >= :cartValue');
		$speditionsRawQuery->orderBy('s.position');
		$speditionsRaw = $speditionsRawQuery->getQuery()->getResult();

		$speditions = $this->fillDao($speditionsRaw);

		return $speditions;
	}

	public function getFirstFreeSpedition()
	{
		$speditionQuery = $this->getEr()->createQueryBuilder('s', 's.id')->andWhere('s.freeFrom > 0')->setMaxResults(1)->orderBy('s.freeFrom', 'ASC');
		$speditionRaw   = $speditionQuery->getQuery()->getOneOrNullResult();

		$firstFreeSpedition = $speditionRaw ? current($this->fillDao([$speditionRaw])) : null;

		return $firstFreeSpedition;
	}

	public function getRawReference($id)
	{
		return $this->getEr()->getReference($id);
	}

	/**
	 * @param int $id
	 *
	 * @return Spedition
	 */
	private function loadRawSpedition($id)
	{
		$speditionRawQuery = $this->getEr()->createQueryBuilder('s', 's.id');
		$speditionRawQuery->andWhere('s.id = :id')->setParameter('id', $id);

		$speditionRaw = $speditionRawQuery->getQuery()->getOneOrNullResult();

		return $speditionRaw;
	}

	/**
	 * @param Spedition[] $speditionsRaw
	 *
	 * @return Dao\Spedition[]
	 */
	protected function fillDao($speditionsRaw)
	{
		$speditions = [];
		foreach ($speditionsRaw as $spedRaw) {
			$spedition = new Dao\Spedition();
			$spedition
				->setId($spedRaw->getId())
				->setIdent($spedRaw->getIdent())
				->setName($spedRaw->getName())
				->setText($spedRaw->getText())
				->setPublished($spedRaw->isPublished)
				->setPosition($spedRaw->getPosition())
				->setPrice((float) $spedRaw->getPrice())
				->setFreeFrom($spedRaw->getFreeFrom())
				->setAvailableTo($spedRaw->getAvailableTo())
				->setImage($spedRaw->image);
			$speditions[$spedition->getId()] = $spedition;
		}

		return $speditions;
	}

}

