<?php declare(strict_types=1);

namespace EshopOrders\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Images\ImageHelper;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\AdminModule\Model\Products;
use EshopOrders\AdminModule\Model\Event\InvoiceRegenerateEvent;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Utils\Image;
use Exception;
use Kdyby\Doctrine\Dql\Join;
use Contributte\EventDispatcher\EventDispatcher;
use Nette\Localization\ITranslator;

/**
 * Class Invoices
 * @package EshopOrders\Model
 *
 * @method Invoice|null get($id)
 */
class Invoices extends BaseEntityService
{
	/** @var string */
	protected $entityClass = Invoice::class;

	/** @var EventDispatcher */
	protected $eventDispatcher;

	/** @var Products */
	protected $products;

	/** @var ITranslator */
	protected $translator;

	/**
	 * Invoices constructor.
	 * @param EventDispatcher $eventDispatcher
	 * @param Products $products
	 * @param ITranslator $translator
	 */
	public function __construct(EventDispatcher $eventDispatcher, Products $products, ITranslator $translator)
	{
		$this->eventDispatcher = $eventDispatcher;
		$this->products = $products;
		$this->translator = $translator;
	}

	/**
	 * @return QueryBuilder
	 */
	public function getQueryBuilder(): QueryBuilder
	{
		return $this->getEr()->createQueryBuilder('i')->orderBy('i.ident', 'desc');
	}

	/**
	 * @param int $orderId
	 * @return Invoice|null
	 * @throws NonUniqueResultException
	 */
	public function getOneByOrder(int $orderId): ?Invoice
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('i')
		   ->from(Invoice::class, 'i')
		   ->join('i.order', 'o', Join::WITH, 'o.invoice = i.id')
		   ->where('o.id = :orderId')
		   ->setParameter('orderId', $orderId);

		return $qb->getQuery()->getOneOrNullResult();
	}

	/**
	 * @return int
	 * @throws NonUniqueResultException
	 */
	public function getIvoicesCount(): int
	{
		$qb = $this->getQueryBuilder();
		return (int) $qb->select($qb->expr()->count('i.id'))->getQuery()->getSingleScalarResult();
	}

	/**
	 * @param int $invoiceId
	 * @return bool
	 */
	public function regenerateInvoice(int $invoiceId): bool
	{
		try {
			$this->em->beginTransaction();

			$qb = $this->em->createQueryBuilder();
			$qb->select('o')
			   ->from(Order::class, 'o')
			   ->join('o.invoice', 'i')
			   ->where('i.id = :invoiceId')
			   ->setParameter('invoiceId', $invoiceId);

			/** @var Order|null $order */
			$order = $qb->getQuery()->getOneOrNullResult();

			if ($order === null) {
				return false;
			}

			$invoice = $order->getInvoice();

			if ($invoice === null) {
				return false;
			}

			// --------- from here the data editing begins ---------

			$invoiceData = $invoice->invoiceData;

			// invoice payment
			$invoicePayment = $invoiceData->payment;
			$invoicePayment->name = $order->getPayment()->getName();
			$invoicePayment->price = $order->getPayment()->getPrice();

			// invoice spedition
			$invoiceSpedition = $invoiceData->spedition;
			$invoiceSpedition->name = $order->getSpedition()->getName();
			$invoiceSpedition->price = $order->getSpedition()->getPrice();

			// customer address
			$orderCustomerAddress = $order->getAddressInvoice();
			$customerAddress = $invoiceData->customer->address;
			$customerAddress->city = $orderCustomerAddress->getCity();
			$customerAddress->country = $orderCustomerAddress->getCountry() ? $orderCustomerAddress->getCountry()->getName() : null;
			$customerAddress->postal = $orderCustomerAddress->getPostal();
			$customerAddress->street = $orderCustomerAddress->getStreet();

			// customer
			$invoiceCustomer = $invoiceData->customer;
			$invoiceCustomer->phone = $orderCustomerAddress->phone;
			$invoiceCustomer->email = $orderCustomerAddress->email;
			$invoiceCustomer->company = $orderCustomerAddress->company;
			$invoiceCustomer->firstName = $orderCustomerAddress->firstName;
			$invoiceCustomer->lastName = $orderCustomerAddress->lastName;
			$invoiceCustomer->idNumber = $orderCustomerAddress->idNumber;
			$invoiceCustomer->vatNumber = $orderCustomerAddress->vatNumber;

			// bank
			$bank = $invoiceData->getSupplier()->getBank();
			$bank->bankAccount = EshopOrdersConfig::load('invoice.bank.bankAccount');
			$bank->bankCode = EshopOrdersConfig::load('invoice.bank.bankCode');
			$bank->iban = EshopOrdersConfig::load('invoice.bank.iban');
			$bank->swift = EshopOrdersConfig::load('invoice.bank.swift');

			// supplier address
			$supplierAddress = $invoiceData->getSupplier()->getAddress();
			$supplierAddress->street = EshopOrdersConfig::load('invoice.supplier.street');
			$supplierAddress->postal = EshopOrdersConfig::load('invoice.supplier.postCode');
			$supplierAddress->city = EshopOrdersConfig::load('invoice.supplier.city');

			// supplier
			$supplier = $invoiceData->getSupplier();
			$supplier->email = EshopOrdersConfig::load('invoice.supplier.email');
			$supplier->vatNumber = EshopOrdersConfig::load('invoice.supplier.taxIdentificationNumber');
			$supplier->idNumber = EshopOrdersConfig::load('invoice.supplier.companyIdentificationNumber');
			$supplier->name = EshopOrdersConfig::load('invoice.supplier.name');
			$supplier->isPayerVat = EshopOrdersConfig::load('invoice.isPayerVat');

			// remove products
			foreach ($invoiceData->products as $product) {
				$this->em->remove($product);
			}

			// actual products
			$invoiceProducts = [];
			foreach ($order->getOrderItems() as $orderItem) {
				$dbProduct = $this->products->get($orderItem->getProductId());
				$invoiceProduct = new Invoice\Product($invoiceData);

				if ($dbProduct->gallery && $dbProduct->gallery->getCoverImage() && file_exists($dbProduct->gallery->getCoverImage()->getFile())) {
					$invoiceProduct->imageBase64 = ImageHelper::createBase64Miniature(
						$dbProduct->gallery->getCoverImage()->getFile(),
						EshopOrdersConfig::load('invoice.productMiniature.width'),
						null,
						\Nette\Utils\Image::SHRINK_ONLY
					);
				}

				$invoiceProduct->name = $dbProduct->getProductText($invoiceData->lang)->name ?? null;
				$invoiceProduct->productId = $dbProduct->getId();
				$invoiceProduct->vatRate = $orderItem->getVatRate();
				$invoiceProduct->price = $orderItem->getPrice();
				$invoiceProduct->vatRate = (int) $orderItem->getVatRate();
				$invoiceProduct->quantity = $orderItem->getQuantity();

				$invoiceProducts[] = $invoiceProduct;
			}

			$invoiceData->products = $invoiceProducts;

			// remove discounts
			foreach ($invoiceData->discounts as $discount) {
				$this->em->remove($discount);
			}

			// actual discounts
			$invoiceDiscounts = [];
			$currentPrice = $invoiceData->getPriceItems();
			foreach ($order->getOrderDiscounts() as $orderDiscount) {
				$amount = $orderDiscount->calculateDiscount($currentPrice);
				$currentPrice += $amount;
				$invoiceDiscount = new Invoice\Discount($invoiceData);
				$invoiceDiscount->price = $amount;
				$invoiceDiscount->name = $orderDiscount->getName();
				$invoiceDiscount->type = $orderDiscount->getType();
				$invoiceDiscount->value = $orderDiscount->getValue();
				$invoiceDiscount->code = $orderDiscount->getCode();

				$invoiceDiscounts[] = $invoiceDiscount;
			}

			$invoiceData->discounts = $invoiceDiscounts;

			$this->eventDispatcher->dispatch('eshopOrders.beforeSaveRegeneratedInvoice', new InvoiceRegenerateEvent($invoice));

			$this->em->persist($invoiceData);
			$this->em->flush();

			$this->em->commit();
			return true;
		} catch (Exception $exception) {
			$this->em->rollback();
			return false;
		}
	}

}
