<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Customer;

use Core\Model\Countries;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Model\Payments;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\Entities\GroupCustomers;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\AdminModule\Model\Customers;

class CustomerForm extends BaseControl
{
	/** @var Customer|null */
	public ?Customer $customer = null;

	/** @var Customers */
	protected Customers $customersService;

	/** @var Countries */
	protected Countries $countries;

	/** @var Payments */
	protected Payments $payments;

	/** @var Speditions */
	protected Speditions $speditions;

	public function __construct(Customers $customers, Countries $countries, Payments $payments, Speditions $speditions)
	{
		$this->customersService = $customers;
		$this->countries        = $countries;
		$this->payments         = $payments;
		$this->speditions       = $speditions;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$groups = ['' => ''];
		foreach ($this->em->getRepository(GroupCustomers::class)->findPairs([], 'name', ['name'], 'id') as $k => $v)
			$groups[$k] = $v;

		$countries = $this->countries->getAllNameColumn();

		$form->addText('name', 'default.firstName', 'user.name')->setRequired()->setMaxLength(255);
		$form->addText('lastname', 'default.lastName', 'user.lastname')->setRequired()->setMaxLength(255);
		$form->addText('email', 'default.email')->setRequired()->setMaxLength(255);
		$form->addText('phone', 'default.phone')->setMaxLength(255);
		$form->addSelect('group', 'eshopOrders.customerForm.group', $groups);

		$form->addCheckboxList('disabledPayments', 'eshopOrders.customer.disabledPayments', $this->payments->getForSelectOption());
		$form->addCheckboxList('disabledSpeditions', 'eshopOrders.customer.disabledSpeditions', $this->speditions->getForSelectOption());

		$addressDelivery = $form->addContainer('addressDelivery');
		$addressInvoice  = $form->addContainer('addressInvoice');

		foreach (['firstName', 'lastName', 'email', 'phone', 'company', 'street', 'city', 'postal', 'idNumber', 'vatNumber', 'country'] as $v) {
			foreach (['addressDelivery', 'addressInvoice'] as $com)
				if ($v == 'email')
					${$com}->addEmail($v, 'eshopOrders.default.' . $v)
						->setMaxLength(255);
				else if ($v == 'country')
					${$com}->addSelect($v, 'eshopOrders.default.' . $v, $countries);
				else
					${$com}->addText($v, 'eshopOrders.default.' . $v)
						->setMaxLength(255);
		}

		$form->addBool('isActive', 'default.isActive')->setRequired();
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if (!$this->customer) {
				$this->getPresenter()->flashMessageWarning('eshopOrders.customerForm.noCreate');

				return false;
			}

			$customer     = $this->customer;
			$user         = $customer->getUser();
			$flashMessage = 'eshopOrders.customerForm.edited';

			$customer->setPhone($values->phone);
			$user->setName($values->name);
			$user->setLastname($values->lastname);
			$user->setEmail($values->email);
			if ($values->group) {
				$customer->setGroupCustomers($this->em->getReference(GroupCustomers::class, $values->group));
			} else {
				$customer->removeGroupCustomers();
			}

			if ($this->customersService->isAdmin($customer->getId()) && $values->isActive != $user->isActive()) {
				$this->getPresenter()->flashMessageWarning('eshopOrders.customerForm.noAdminDeactivate');
			} else {
				$user->disable(!$values->isActive);
			}

			$addressDelivery = $customer->getAddressDelivery() ?: new CustomerAddress($customer);
			$addressInvoince = $customer->getAddressInvoice() ?: new CustomerAddress($customer);

			$customer->setAddressDelivery($addressDelivery);
			$customer->setAddressInvoice($addressInvoince);

			foreach (['firstName', 'lastName', 'email', 'phone', 'company', 'street', 'city', 'postal', 'idNumber', 'vatNumber', 'country'] as $v) {
				foreach (['addressDelivery', 'addressInvoice'] as $com) {
					if ($v == 'country') {
						if ($values->$com->$v)
							$customer->$com->$v = $this->countries->getReference($values->$com->$v);
					} else {
						$customer->$com->$v = $values->$com->$v;
					}
				}
			}

			$currentDisabledPayments = $customer->disabledPayments->toArray();
			foreach ($values->disabledPayments as $id) {
				if (!isset($currentDisabledPayments[$id]))
					$customer->disabledPayments->set($id, $this->payments->getReference($id));

				unset($currentDisabledPayments[$id]);
			}

			foreach ($currentDisabledPayments as $entity)
				$customer->disabledPayments->remove($entity->getId());

			$currentDisabledSpeditions = $customer->disabledSpeditions->toArray();
			foreach ($values->disabledSpeditions as $id) {
				if (!isset($currentDisabledSpeditions[$id]))
					$customer->disabledSpeditions->set($id, $this->speditions->getReference($id));

				unset($currentDisabledSpeditions[$id]);
			}

			foreach ($currentDisabledSpeditions as $entity)
				$customer->disabledSpeditions->remove($entity->getId());

			$this->em->persist($addressDelivery);
			$this->em->persist($addressInvoince);

			$this->em->persist($customer);
			$this->em->persist($user);

			$this->em->flush();
			$form->addCustomData('customerId', $customer->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setCustomer($id)
	{
		$this->customer = $customer = $this->customersService->get($id);

		if ($customer) {
			$o = $customer;
			$u = $o->getUser();

			$d = [
				'phone'              => $o->getPhone(),
				'name'               => $u->getName(),
				'lastname'           => $u->getLastname(),
				'email'              => $u->getEmail(),
				'isActive'           => $u->isActive(),
				'disabledPayments'   => $o->disabledPayments->getKeys(),
				'disabledSpeditions' => $o->disabledSpeditions->getKeys(),
			];

			foreach (['firstName', 'lastName', 'email', 'phone', 'company', 'street', 'city', 'postal', 'idNumber', 'vatNumber', 'country'] as $v) {
				foreach (['addressDelivery', 'addressInvoice'] as $com) {
					if (!$customer->$com)
						continue;

					if ($v == 'country') {
						$d[$com][$v] = $customer->$com->$v ? $customer->$com->$v->getId() : null;
					} else {
						$d[$com][$v] = $customer->$com->$v;
					}
				}
			}

			$this['form']->setDefaults($d);

			if ($o->getGroupCustomers() && array_key_exists($o->getGroupCustomers()->getId(), $this['form']['group']->getItems()))
				$this['form']['group']->setDefaultValue($o->getGroupCustomers()->getId());
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
