<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Customer;

use Core\Model\UI\BaseControl;
use Doctrine\ORM\QueryBuilder;
use EshopOrders\AdminModule\Model\Customers;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\GroupCustomers;

class CustomersGrid extends BaseControl
{
	/** var Customers */
	protected $customersService;

	public function __construct(Customers $customers)
	{
		$this->customersService = $customers;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setStrictSessionFilterValues(false);

		$qb = $this->customersService->getEr()->createQueryBuilder('p')
			->addSelect('user, adDel, adInv, grp')
			->innerJoin('p.user', 'user')
			->leftJoin('p.addressDelivery', 'adDel')
			->leftJoin('p.addressInvoice', 'adInv')
			->leftJoin('p.groupCustomers', 'grp')
			->orderBy('p.id');
		$grid->setDataSource($qb);

		$groupCustomers = [];
		foreach ($this->em->getRepository(GroupCustomers::class)->createQueryBuilder('gc')
			         ->select('gc.id, gc.name')->getQuery()->getArrayResult() as $gc)
			$groupCustomers[$gc['id']] = $gc['name'];

		//Columns
		$grid->addColumnLink('name', 'default.firstName', 'Customers:editCustomer', 'user.name');
		$grid->addColumnLink('lastname', 'default.lastName', 'Customers:editCustomer', 'user.lastname');
		$grid->addColumnEmail('email', 'default.email', 'user.email');
		$grid->addColumnPhone('phone', 'default.phone', 'phone', 'addressInvoice.country.id', true);
		$grid->addColumnText('group', 'default.group')->setRenderer(function(Customer $row) {
			return $row->getGroupCustomers() ? $row->getGroupCustomers()->name : '';
		});
		$grid->addColumnStatus('isActive', 'default.isActive', 'user.isActive')->setAlign('center')
			->addOption(1, 'eshopOrders.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopOrders.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter
		$grid->addFilterText('name', '', 'user.name');
		$grid->addFilterText('lastname', '', 'user.lastname');
		$grid->addFilterText('email', '', 'user.email');
		$grid->addFilterText('phone', '', 'adDel.phone');
		$grid->addFilterSelect('group', '', [
				null => '',
				'no' => $this->t('eshopOrders.customer.withoutGroup'),
			] + $groupCustomers, 'grp.id')
			->setCondition(function(QueryBuilder $qb, $value) {
				if ($value === 'no')
					$qb->andWhere('grp.id IS NULL');
				else
					$qb->andWhere('grp.id = :customerGroup')->setParameter('customerGroup', $value);
			});
		$grid->addFilterSelect('isActive', '', [
			null => '',
			1    => $this->t('default.yes'),
			0    => $this->t('default.no'),
		], 'user.isActive');

		// Actions
		$grid->addAction('edit', '', 'Customers:editCustomer')->setIcon('edit')->setBsType('primary');


		$grid->addGroupSelectAction('eshopOrders.defaultGrid.editGroupCustomers',
			[0 => $this->t('eshopOrders.defaultGrid.nullGroupCustomers')] + $groupCustomers)
			->onSelect[] = [$this, 'gridEditGroupCustomers'];

		return $grid;
	}


	/*******************************************************************************************************************
	 * ==================  Handle
	 */


	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->customersService->isAdmin($id)) {
			$presenter->flashMessageWarning('eshopOrders.customerForm.noAdminDeactivate');
		} else {
			if ($this->customersService->setPublish($id, $newStatus)) {
				$presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
			} else {
				$presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');
			}
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/**
	 * @param array $ids
	 * @param       $groupCustomers
	 *
	 * @throws \Nette\Application\AbortException
	 */
	public function gridEditGroupCustomers(array $ids, $groupCustomers)
	{
		$presenter = $this->getPresenter();
		if ($this->customersService->setGroupCustomers($ids, $groupCustomers)) {
			$presenter->flashMessageSuccess('eshopOrders.defaultGrid.groupCustomersChanged');
		} else {
			$presenter->flashMessageDanger('eshopOrders.defaultGrid.groupCustomersChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

}
