<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Expedition;

use Core\Model\UI\BaseControl;
use EshopOrders\AdminModule\Model\Orders;
use EshopOrders\AdminModule\Model\Statuses;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Application\LinkGenerator;

class OrdersGrid extends BaseControl
{
	/** @var Orders */
	protected Orders $orders;

	/** @var LinkGenerator */
	protected LinkGenerator $linkGenerator;

	/** @var Statuses */
	protected Statuses $statusService;

	public function __construct(Orders $orders, LinkGenerator $linkGenerator, Statuses $statuses)
	{
		$this->orders        = $orders;
		$this->linkGenerator = $linkGenerator;
		$this->statusService = $statuses;
	}

	public function render(): void
	{
		$this->template->orders = $this->orders->getOrdersForExpedition();
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ====== Handle
	 */

	/**
	 * @param int $orderId
	 */
	public function handleProcessOrder(int $orderId): void
	{
		$presenter = $this->getPresenter();

		// Export
		$exportResult = $this->orders->sendExpeditionData([$orderId]);
		$exportErrors = $exportResult['errorCount'];
		$exportOks    = $exportErrors['okCount'];

		if ($exportErrors > 0)
			$presenter->flashMessageDanger('eshopOrders.expeditionOrdersGrid.messages.exportError');
		else if ($exportErrors === 0 && $exportOks > 0)
			$presenter->flashMessageSuccess('eshopOrders.expeditionOrdersGrid.messages.exported');

		// Label
		$labelResult = $this->orders->generateLabel([$orderId]);
		$labelFiles  = $labelResult['files'];

		if ($labelFiles) {
			$file = $labelFiles[0];

			$arr                                = [
				'name' => 'Label',
				'url'  => $this->linkGenerator->link('Core:Admin:FileResponse:', [base64_encode(serialize([
					'file'        => $file,
					'filename'    => basename($file),
					'download'    => false,
					'contentType' => 'application/pdf',
				]))]),
			];
			$presenter->payload->fileRequests[] = $arr;
			$presenter->flashMessageSuccess('eshopOrders.expeditionOrdersGrid.messages.labelGenerated');

			if ($this->statusService->changeStatus([$orderId], EshopOrdersConfig::load('expeditionOrdersGrid.statusAfterExport', 'spedition')))
				$presenter->flashMessageSuccess('eshopOrders.order.statusChange');
			else
				$presenter->flashMessageDanger('eshopOrders.order.statusChangeFailed');
		} else {
			$presenter->flashMessageDanger('eshopOrders.expeditionOrdersGrid.messages.labelGenerateError');
		}

		$presenter->redrawControl('flashes');
		$this->redrawControl('table');
	}
}
