<?php declare(strict_types=1);

namespace EshopOrders\AdminModule\Components\Invoice;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\NumericalSeries;
use EshopOrders\Model\Entities\SiteInvoiceConfig;
use EshopOrders\Model\InvoiceConfigRepository;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Utils\Strings;
use Exception;
use Nette\Forms\IControl;
use Nette\Utils\ArrayHash;
use ReflectionException;

class InvoiceConfigForm extends BaseControl
{
	/** @var InvoiceConfigRepository */
	protected $invoiceConfigRepository;

	/** @var Invoices */
	protected $invoices;

	/** @var InvoiceConfig|null */
	protected $invoiceConfig;

	/** @var Sites */
	protected $sites;

	/**
	 * InvoiceConfigForm constructor.
	 * @param int|null $id
	 * @param InvoiceConfigRepository $invoiceConfigRepository
	 * @param Invoices $invoices
	 * @param Sites $sites
	 */
	public function __construct(?int $id, InvoiceConfigRepository $invoiceConfigRepository, Invoices $invoices, Sites $sites)
	{
		$this->invoiceConfig = $id ? $invoiceConfigRepository->get($id) : null;
		$this->invoiceConfigRepository = $invoiceConfigRepository;
		$this->invoices = $invoices;
		$this->sites = $sites;
	}

	/**
	 * @return BaseForm
	 */
	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$sites = $this->sites->getOptionsForSelect();

		$form->addCheckboxList('sites', 'eshopOrders.invoiceConfigsGrid.sites', $sites)
			->setRequired()
			->setDisabled(array_map(static function (SiteInvoiceConfig $config) {
				return $config->site->getIdent();
			}, $this->invoiceConfigRepository->getUsed()));

		$form->addInteger('maturity', 'eshopOrders.invoiceConfigForm.maturity')
			 ->setRequired()
			 ->addRule($form::MIN, null, 1);

		$form->addGroup('eshopOrders.invoiceConfigForm.numericalSeries.default.caption');

		$form->addText('prefix', 'eshopOrders.invoiceConfigForm.numericalSeries.default.prefix')
			 ->setRequired($this->invoiceConfig ? true : false)
			 ->addRule(static function (IControl $control): bool { // If prefix contains month wildcards then it must also contain year wildcards
			 	$value = $control->getValue();
			 	return !(Strings::containsMore($value, NumericalSeries::POSSIBLE_PREFIX_WILDCARDS['month']) &&
					!Strings::containsMore($value, NumericalSeries::POSSIBLE_PREFIX_WILDCARDS['year']));
			 }, 'eshopOrders.invoiceConfigForm.numericalSeries.errors.prefixValue');

		$form->addInteger('digitsCount', 'eshopOrders.invoiceConfigForm.numericalSeries.default.digitsCount')
			 ->setRequired($this->invoiceConfig ? true : false)
			 ->addRule($form::MIN, null, 1);

		$form->addInteger('startNumber', 'eshopOrders.invoiceConfigForm.numericalSeries.default.startNumber')
			 ->setRequired($this->invoiceConfig ? true : false);

		$form->addHidden('id');

		$form->addSubmit('submit', 'default.save');

		$form->onAnchor[] = function (BaseForm $form) {

			// we cannot change numerical series if invoices count is bigger than 0
			if ($this->invoiceConfig && $this->invoices->getIvoicesCount($this->getSitesIdent()) > 0) {
				$form['prefix']->setDisabled();
				$form['digitsCount']->setDisabled();
				$form['startNumber']->setDisabled();
			}
		};

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	/**
	 * @param BaseForm $form
	 * @param ArrayHash $values
	 * @throws Exception
	 */
	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		if (count($values->sites) === 0) {
			$form->addError('eshopOrders.invoiceConfigForm.numericalSeries.errors.sitesFilled');
		}

		$charsLength = 10;
		if ((Strings::length($values->prefix) + $values->digitsCount) > $charsLength) { // prefix + digits count char length must not exceed 10, becouse variable symbol is up to 10 chars
			$form->addError($this->t('eshopOrders.invoiceConfigForm.numericalSeries.errors.charsLength', null, ['length' => $charsLength]));
		} else if (Strings::length($values->startNumber) > $values->digitsCount) {
			$form->addError('eshopOrders.invoiceConfigForm.numericalSeries.errors.maxStartNumber');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('formErrors');
		}
	}

	/**
	 * @param BaseForm $form
	 * @param ArrayHash $values
	 * @throws Exception
	 */
	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		if ($values->id) {
			$this->invoiceConfig = $this->invoiceConfigRepository->get((int) $values->id);
		}

		if ($this->invoiceConfig === null) {
			$numericalSeries = new NumericalSeries($values->prefix, $values->digitsCount, $values->startNumber);
			$this->invoiceConfig = new InvoiceConfig($values->maturity, $numericalSeries);
			$this->invoiceConfig->numericalSeries = $numericalSeries;

			foreach ($values->sites as $siteIdent) {
				$site = $this->sites->get($siteIdent);
				if (!$site) {
					return;
				}
				$this->invoiceConfig->siteInvoiceConfigs[] = new SiteInvoiceConfig($this->invoiceConfig, $site);
			}
		} else {
			$defaultSiteInvoiceConfigs = $this->invoiceConfig->siteInvoiceConfigs->toArray();
			$defaultSites = array_map(function (SiteInvoiceConfig $config) {
				return $config->site->getIdent();
			}, $defaultSiteInvoiceConfigs);

			$newSites = $values->sites;

			// remove
			foreach (array_diff($defaultSites, $newSites) as $siteIdent) {
				foreach ($defaultSiteInvoiceConfigs as $invoiceSiteConfig) {
					if ($invoiceSiteConfig->site->getIdent() === $siteIdent) {
						$this->invoiceConfig->siteInvoiceConfigs->removeElement($invoiceSiteConfig);
						$this->em->remove($invoiceSiteConfig);
						break;
					}
				}
			}

			// add
			foreach (array_diff($newSites, $defaultSites) as $siteIdent) {
				$site = $this->sites->get($siteIdent);
				if ($site) {
					$this->invoiceConfig->siteInvoiceConfigs->add(new SiteInvoiceConfig($this->invoiceConfig, $site));
				}
			}
		}

		$this->invoiceConfig->maturity = $values->maturity;
		if (!$values->id) {
			$this->invoiceConfig->numericalSeries->prefix = $values->prefix ?? $this->invoiceConfig->numericalSeries->prefix;
			$this->invoiceConfig->numericalSeries->startNumber = $values->startNumber ?? $this->invoiceConfig->numericalSeries->startNumber;
			$this->invoiceConfig->numericalSeries->digitsCount = $values->digitsCount ?? $this->invoiceConfig->numericalSeries->digitsCount;
		}

		$this->em->persist($this->invoiceConfig);
		$this->em->flush();
		$this->presenter->flashMessageSuccess('default.saved');
		$this->presenter->redrawControl('flashes');
	}

	/**
	 * @throws ReflectionException
	 */
	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @throws Exception
	 */
	public function setDefaults(): void
	{
		$defaultConfig = $this->invoiceConfig;

		$arr = array_map(static function (SiteInvoiceConfig $config) {
			return $config->site->getIdent();
		}, $defaultConfig->siteInvoiceConfigs->toArray());

		$form = $this['form'];
		$form->setDefaults([
			'maturity' => $defaultConfig->maturity,
			'prefix' => $defaultConfig->numericalSeries->prefix,
			'digitsCount'=> $defaultConfig->numericalSeries->digitsCount,
			'startNumber'=> $defaultConfig->numericalSeries->startNumber,
			'sites' => $arr,
			'id' => $defaultConfig->getId()
		]);
		$form['sites']->setDisabled(array_filter($this->getSitesIdent(), static function (string $siteIdent) use ($arr): bool {
			return !in_array($siteIdent, $arr, true);
		}));
	}

	/**
	 * @return array
	 */
	public function getSitesIdent(): array
	{
		return array_map(static function (SiteInvoiceConfig $config): string {
			return $config->site->getIdent();
		}, $this->invoiceConfigRepository->getUsed());
	}

}
