<?php declare(strict_types=1);

namespace EshopOrders\AdminModule\Components\Invoice;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\SiteInvoiceConfig;
use EshopOrders\Model\InvoiceConfigRepository;
use EshopOrders\Model\Invoices;
use ReflectionException;
use Ublaboo\DataGrid\Exception\DataGridException;

class InvoiceConfigsGrid extends BaseControl
{
	/** @var InvoiceConfigRepository */
	protected $invoiceConfigs;

	/** @var Invoices */
	protected $invoices;

	/**
	 * InvoiceConfigsGrid constructor.
	 * @param InvoiceConfigRepository $invoiceConfigs
	 * @param Invoices $invoices
	 */
	public function __construct(InvoiceConfigRepository $invoiceConfigs, Invoices $invoices)
	{
		$this->invoiceConfigs = $invoiceConfigs;
		$this->invoices = $invoices;
	}

	/**
	 * @return BaseDataGrid
	 * @throws DataGridException
	 */
	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDataSource($this->invoiceConfigs->getQueryBuilder());

		$grid->addColumnText('sites', 'eshopOrders.invoiceConfigsGrid.sites')
			->setRenderer(static function (InvoiceConfig $config) {
				return implode(', ', array_map(static function (SiteInvoiceConfig $config) {
					return $config->site->getIdent();
				}, $config->siteInvoiceConfigs->toArray()));
			});

		$grid->addColumnText('maturity', 'eshopOrders.invoiceConfigsGrid.maturity');
		$grid->addColumnText('prefix', 'eshopOrders.invoiceConfigsGrid.numericalSeries.default.prefix')
			->setRenderer(static function (InvoiceConfig $config) {
				return $config->numericalSeries->prefix;
			});
		$grid->addColumnText('digitsCount', 'eshopOrders.invoiceConfigsGrid.numericalSeries.default.digitsCount')
			->setRenderer(static function (InvoiceConfig $config) {
				return $config->numericalSeries->digitsCount;
			});
		$grid->addColumnText('startNumber', 'eshopOrders.invoiceConfigsGrid.numericalSeries.default.startNumber')
			->setRenderer(static function (InvoiceConfig $config) {
				return $config->numericalSeries->startNumber;
			});

		$grid->addAction('edit', '', 'invoiceSettings!', ['itemId' => 'id'])
			->addClass('ajax')
			->setIcon('edit')->setBsType('primary');

		$grid->addAction('remove', '', 'remove!', ['itemId' => 'id'])
			->setRenderCondition(function (InvoiceConfig $config) {
				return $this->invoices->getIvoicesCount($this->getSitesIdent($config->siteInvoiceConfigs->toArray())) === 0;
			})
			->addClass('ajax')
			->setIcon('trash')
			->setBsType('danger')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	/**
	 * @param mixed $itemId
	 */
	public function handleRemove($itemId): void
	{
		$this->invoiceConfigs->remove((int) $itemId);
		$this['grid']->reload();
	}

	/**
	 * @throws ReflectionException
	 */
	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @param SiteInvoiceConfig[] $sites
	 * @return array
	 */
	public function getSitesIdent(array $sites): array
	{
		return array_map(static function (SiteInvoiceConfig $config): string {
			return $config->site->getIdent();
		}, $sites);
	}

}
